﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>
#include <memory>

#include "./succinct/FileInputStream.h"
#include "./../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_BinaryReader.h"

#include "NgWordConverter.h"

// Windows 用の AhoCorasick バイナリのヘッダの作成
errno_t NgWordConverter::InitializeAcWinHeader(nlib_fd* pOutWinHdaderFd)
{
    // 既にある場合は削除
    errno_t result = nlib_remove(m_WinHeaderPath.c_str());
    if (result != 0 && result != ENOENT)
    {
        nlib_printf("Cannot remove %s\n", m_WinHeaderPath.c_str());
        return result;
    }
    result = nlib_fd_creat(pOutWinHdaderFd, m_WinHeaderPath.c_str(), 0);
    if (result != 0)
    {
        nlib_printf("Cannot create %s\n", m_WinHeaderPath.c_str());
        return result;
    }
    // Copyright の記述
    size_t readSize;
    size_t writeSize;
    {
        nlib_fd headerFd;
        result = nlib_fd_open(&headerFd, m_CopyrightPath.c_str(), NLIB_FD_O_RDONLY);
        if (result != 0)
        {
            nlib_printf("Cannot open %s\n", m_CopyrightPath.c_str());
            return result;
        }
        nlib_offset headerSizeTmp;
        result = nlib_fd_getsize(&headerSizeTmp, headerFd);
        if (result != 0)
        {
            nlib_printf("Cannot get file size %s\n", m_CopyrightPath.c_str());
            return result;
        }
        assert(headerSizeTmp > 0);
        size_t headerSize = static_cast<size_t>(headerSizeTmp);
        std::unique_ptr<char[]> pHeader(new (std::nothrow) char[headerSize]);
        if (!pHeader)
        {
            return ENOMEM;
        }
        result = nlib_fd_read(&readSize, headerFd, pHeader.get(), headerSize);
        if (result != 0)
        {
            nlib_printf("Cannot read %s\n", m_CopyrightPath.c_str());
            return result;
        }
        assert(readSize == headerSize);

        nlib_fd_write(&writeSize, *pOutWinHdaderFd, pHeader.get(), headerSize);
        std::string nameSpace = "namespace nn { namespace ngc { namespace detail {\r\n\r\n";
        result = nlib_fd_write(&writeSize, *pOutWinHdaderFd, nameSpace.c_str(), nameSpace.size());
        if (result != 0)
        {
            nlib_printf("Cannot write %s\n", m_WinHeaderPath.c_str());
            return result;
        }

        result = nlib_fd_flush(*pOutWinHdaderFd);
        if (result != 0)
        {
            nlib_printf("Cannot flush %s\n", m_WinHeaderPath.c_str());
            return result;
        }
        result = nlib_fd_close(headerFd);
        if (result != 0)
        {
            nlib_printf("Cannot close %s\n", m_CopyrightPath.c_str());
            return result;
        }
    }

    return 0;
}

// Windows 用の AhoCorasick バイナリのヘッダの作成
errno_t NgWordConverter::FinalizeAcWinHeader(nlib_fd* pOutWinHdaderFd)
{
    errno_t result = 0;
    size_t writeSize = 0;

    result = nlib_fd_write(&writeSize, *pOutWinHdaderFd, "}}} // nn::ngc::detail\r\n", nlib_strnlen("}}} // nn::ngc::detail\r\n", 32));
    if (result != 0)
    {
        nlib_printf("Cannot write file\n");
        return result;
    }

    result = nlib_fd_close(*pOutWinHdaderFd);
    if (result != 0)
    {
        nlib_printf("Cannot close file\n");
        return result;
    }

    pOutWinHdaderFd = 0;
    return 0;
}

// Windows 用の AhoCorasick バイナリのヘッダへの書き込み
errno_t NgWordConverter::WriteAcWinHeader(std::string fileName, nlib_fd winHeaderFd, const std::string* pInputPath)
{
    const std::string pAcType[m_AcTypeSize] = { "NotB", "B1", "B2" };
    const std::string pAcLanguage[m_LanguageSize + 1] = { "Japanese", "AmericanEnglish", "CanadianFrench",
        "LatinAmericanSpanish", "BritishEnglish", "French", "German", "Italian",
        "Spanish", "Dutch", "Korean", "Chinese", "Portuguese", "Russian",
        "SouthAmericanPortuguese", "Taiwanese", "Common" };
    errno_t result = 0;

    for (int i = 0; i < m_AcTypeSize; ++i)
    {
        // バイナリの読み込み
        nn::ngc::detail::BinaryReader reader;
        nn::ngc::detail::FileInputStream inputStream;
        result = inputStream.Init();
        if (result != 0)
        {
            nlib_printf("Cannot initialize FileInputStream\n");
            return result;
        }
        result = inputStream.Open(pInputPath[i].c_str());
        if (result != 0)
        {
            nlib_printf("Cannot open %s\n", pInputPath[i].c_str());
            return result;
        }
        result = reader.Init();
        if (result != 0)
        {
            nlib_printf("Cannot initialize BinaryReader\n");
            return result;
        }
        result = reader.Open(&inputStream);
        if (result != 0)
        {
            nlib_printf("Cannot open FileInputStream\n");
            return result;
        }

        // 配列宣言部分の記述
        int index = (fileName == "common") ? m_LanguageSize : std::stoi(fileName);
        // nlib_fd_write は unix スタイルで記述するので改行は \r\n と記述(CR+LF)
        size_t writeSize = 0;
        std::string firstLine = "unsigned char pAc" + pAcLanguage[index] + pAcType[i] + "[] = {\r\n";
        nlib_fd_write(&writeSize, winHeaderFd, firstLine.c_str(), firstLine.size());
        nlib_fd_write(&writeSize, winHeaderFd, "    ", 4);

        // 各配列内の値の記述
        unsigned char p;
        int lfCount = 0;
        size_t count;
        if (reader.Read(&p))
        {
            result = nlib_dprintf(winHeaderFd, &count, "0x%02X", p);
            while (reader.Read(&p))
            {
                result = nlib_fd_write(&writeSize, winHeaderFd, ",", 1);
                if (result != 0)
                {
                    nlib_printf("Cannot write file\n");
                    return result;
                }
                lfCount++;
                // 適度に改行
                if (lfCount % 16 == 0)
                {
                    result = nlib_fd_write(&writeSize, winHeaderFd, "\r\n   ", nlib_strnlen("\r\n   ", 8));
                    lfCount = 0;
                }
                result = nlib_dprintf(winHeaderFd, &count, " 0x%02X", p);
            }
        }
        result = nlib_fd_write(&writeSize, winHeaderFd, " };\r\n\r\n", nlib_strnlen(" };\r\n\r\n", 8));
        if (result != 0)
        {
            nlib_printf("Cannot write file\n");
            return result;
        }
    }

    result = nlib_fd_flush(winHeaderFd);
    if (result != 0)
    {
        nlib_printf("Cannot flush file\n");
        return result;
    }

    return 0;
}
