﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cassert>

#include <nn/nlib/Config.h>

#include <nn/util/util_Compression.h>

#include "NgWordConverter.h"

// エクスポート時に利用するファイルパスが存在するか確認
bool NgWordConverter::CheckDirectoryPath()
{
    int result = 0;
    int retVal = 0;

    retVal = nlib_exist_path(&result, m_RootPath.c_str());
    if (result == 0 || retVal != 0)
    {
        nlib_printf("%s does not exist\n", m_RootPath.c_str());
        return false;
    }
    retVal = nlib_exist_path(&result, m_NgResourceInputPath.c_str());
    if (result == 0 || retVal != 0)
    {
        nlib_printf("%s does not exist\n", m_NgResourceInputPath.c_str());
        return false;
    }
    retVal = nlib_exist_path(&result, m_NgResourceOutputPath.c_str());
    if (result == 0 || retVal != 0)
    {
        nlib_printf("%s does not exist\n", m_NgResourceOutputPath.c_str());
        return false;
    }
    retVal = nlib_exist_path(&result, m_CopyrightPath.c_str());
    if (result == 0 || retVal != 0)
    {
        nlib_printf("%s does not exist\n", m_CopyrightPath.c_str());
        return false;
    }
    std::string winHeaderDirPath = m_WinHeaderPath;
    while (winHeaderDirPath[winHeaderDirPath.length() - 1] != '\\' &&
        winHeaderDirPath[winHeaderDirPath.length() - 1] != '/')
    {
        winHeaderDirPath.pop_back();
    }
    retVal = nlib_exist_path(&result, winHeaderDirPath.c_str());
    if (result == 0 || retVal != 0)
    {
        nlib_printf("%s does not exist\n", winHeaderDirPath.c_str());
        return false;
    }

    return true;
}

// 生成した中間ファイルの削除
errno_t NgWordConverter::DeleteIntermediates(std::string fileName, const std::string* pAcStr)
{
    for (int i = 0; i < m_AcTypeSize; ++i)
    {
        errno_t result = nlib_remove(pAcStr[i].c_str());
        if (result != 0 && result != ENOENT)
        {
            nlib_printf("Cannot remove %s\n", pAcStr[i].c_str());
            return result;
        }
    }
    return 0;
}

// HorizonOS 用の AhoCorasick バイナリを圧縮
errno_t NgWordConverter::CompressAcBinary(std::string fileName, const std::string* pOutputPath, const std::string* pInputPath)
{
    std::unique_ptr<char[]> pWorkBuf(new (std::nothrow) char[nn::util::CompressGzipWorkBufferSizeDefault]);
    if (!pWorkBuf.get())
    {
        return ENOMEM;
    }

    for (int i = 0; i < m_AcTypeSize; ++i)
    {
        nlib_fd outputFd = 0;
        nlib_fd inputFd = 0;

        // 既にファイルがある場合は削除
        errno_t result = nlib_remove(pOutputPath[i].c_str());
        if (result != 0 && result != ENOENT)
        {
            nlib_printf("Cannot remove %s\n", pOutputPath[i].c_str());
            return result;
        }
        result = nlib_fd_creat(&outputFd, pOutputPath[i].c_str(), 0);
        if (result != 0)
        {
            nlib_printf("Cannot create %s\n", pOutputPath[i].c_str());
            return result;
        }
        result = nlib_fd_open(&inputFd, pInputPath[i].c_str(), NLIB_FD_O_RDONLY);
        if (result != 0)
        {
            nlib_printf("Cannot open %s\n", pInputPath[i].c_str());
            return result;
        }
        nlib_offset inputSizeTmp = 0;
        result = nlib_fd_getsize(&inputSizeTmp, inputFd);
        if (result != 0)
        {
            nlib_printf("Cannot get file size %s\n", pInputPath[i].c_str());
            return result;
        }
        assert(inputSizeTmp > 0);
        size_t inputSize = static_cast<size_t>(inputSizeTmp);
        size_t outputSize = 0;
        std::unique_ptr<char[]> pSrc(new (std::nothrow) char[inputSize]);
        std::unique_ptr<char[]> pDst(new (std::nothrow) char[inputSize]);
        if (!pSrc.get() || !pDst.get())
        {
            return ENOMEM;
        }
        size_t readSize = 0;
        result = nlib_fd_read(&readSize, inputFd, pSrc.get(), inputSize);
        if (result != 0)
        {
            nlib_printf("Cannot read %s\n", pInputPath[i].c_str());
            return result;
        }
        assert(readSize == inputSize);
        nn::util::CompressGzip(&outputSize,
                               pDst.get(),
                               inputSize,
                               pSrc.get(),
                               inputSize,
                               pWorkBuf.get(),
                               nn::util::CompressGzipWorkBufferSizeDefault);
        nlib_printf("    CompressGzip: %zu bytes -> %zu bytes (%zu bytes)\n",
            inputSize, outputSize, inputSize - outputSize);
        size_t writeSize = 0;
        nlib_fd_write(&writeSize, outputFd, pDst.get(), outputSize);
        result = nlib_fd_flush(outputFd);
        if (result != 0)
        {
            nlib_printf("Cannot flush %s\n", pOutputPath[i].c_str());
            return result;
        }
        result = nlib_fd_close(inputFd);
        if (result != 0)
        {
            nlib_printf("Cannot close %s\n", pInputPath[i].c_str());
            return result;
        }
        result = nlib_fd_close(outputFd);
        if (result != 0)
        {
            nlib_printf("Cannot close %s\n", pOutputPath[i].c_str());
            return result;
        }
    }

    return 0;
}

// AhoCorasick バイナリの生成と書き出し
errno_t NgWordConverter::CreateAcBinary()
{
    // 利用する各ディレクトリの存在確認
    if (!CheckDirectoryPath())
    {
        return EIO;
    }

    // Windows 用の AhoCorasick バイナリ書き込み先
    nlib_printf("  Create Header for Windows.\n");
    nlib_fd winHeaderFd;
    errno_t result = InitializeAcWinHeader(&winHeaderFd);
    if (result != 0)
    {
        return result;
    }

    for (int i = 0; i <= m_LanguageSize; ++i)
    {
        // 3 つの AhoCorasick を作成する
        // 0: \b を持たないキーワードの AC
        // 1: \b を持つキーワードの AC 1 (\b を抜いてキーワードを格納)
        // 2: \b を持つキーワードの AC 2 (\b を抜かずにキーワードを格納)
        const std::string fileName = (i < m_LanguageSize) ? std::to_string(i) : "common";
        const std::string pAcTmpStr[m_AcTypeSize] = {
            m_NgResourceOutputPath + "ac_" + fileName + "_not_b_nx_tmp",
            m_NgResourceOutputPath + "ac_" + fileName + "_b1_nx_tmp",
            m_NgResourceOutputPath + "ac_" + fileName + "_b2_nx_tmp" };
        const std::string pAcStr[m_AcTypeSize] = {
            m_NgResourceOutputPath + "ac_" + fileName + "_not_b_nx",
            m_NgResourceOutputPath + "ac_" + fileName + "_b1_nx",
            m_NgResourceOutputPath + "ac_" + fileName + "_b2_nx" };

        nlib_printf("  %s Create AhoCorasick Binary\n", fileName.c_str());
        result = CreatePlainAcBinary(fileName, pAcTmpStr);
        if (result != 0)
        {
            return result;
        }
        nlib_printf("  %s Write Binary to Header for Windows\n", fileName.c_str());
        result = WriteAcWinHeader(fileName, winHeaderFd, pAcTmpStr);
        if (result != 0)
        {
            return result;
        }
        nlib_printf("  %s Compress\n", fileName.c_str());
        result = CompressAcBinary(fileName, pAcStr, pAcTmpStr);
        if (result != 0)
        {
            return result;
        }
        // 中間ファイルの削除
        nlib_printf("  %s Delete intermediates\n", fileName.c_str());
        result = DeleteIntermediates(fileName, pAcTmpStr);
        if (result != 0)
        {
            return result;
        }
    }

    result = FinalizeAcWinHeader(&winHeaderFd);
    if (result != 0)
    {
        return result;
    }

    return 0;
}
