﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.LocateVsInstallation
{
    class Program
    {
        enum CheckTarget
        {
            BasicBuild,
            FullDevelopment,
        }

        /// <summary>
        /// Roslyn コンパイラだけを必要とする .NET プログラムのビルドに必要なコンポーネント
        /// </summary>
        // 本当は ExecuteMsBuild 等のパラメータに思われる
        private static IEnumerable<string> ComponentsForBasicDotNetBuild { get; } = new string[]
        {
            "Microsoft.Component.MSBuild",
            "Microsoft.VisualStudio.Component.Roslyn.Compiler",
        };

        /// <summary>
        /// フル機能のビルドを行うのに必要なコンポーネント。我々が必要とするコンポーネント全て。
        /// </summary>
        // 本当は ExecuteMsBuild 等のパラメータに思われる
        private static IEnumerable<string> ComponentsForFullFeatureBuild { get; } = new string[]
        {
            // MSBuild
            "Microsoft.Component.MSBuild",
            "Microsoft.VisualStudio.Component.Roslyn.Compiler",

            // C++ Desktop 開発
            "Microsoft.VisualStudio.Component.VC.Tools.x86.x64",
            "Microsoft.VisualStudio.Component.VC.CLI.Support",
            "Microsoft.VisualStudio.Component.Static.Analysis.Tools",
            "Microsoft.VisualStudio.Component.Windows81SDK",
            "Microsoft.VisualStudio.Component.Windows10SDK.10240",
            "Microsoft.VisualStudio.Component.Windows10SDK.15063.Desktop",

            // Visual Studio 拡張の開発
            "Microsoft.VisualStudio.Component.VSSDK",
        };

        static void Main(string[] args)
        {
            var target = ParseCommandLineOptionOrDie(args);

            if (!TestVisualStudioInstance(target, Nact.Utility.VisualStudioCollection.VisualStudio15ChannelId))
            {
                Console.Error.WriteLine("error");
                Environment.Exit(1);
            }

            var vsInstance = GetVisualStudioInstance(target, Nact.Utility.VisualStudioCollection.VisualStudio15ChannelId);
            Console.Write(vsInstance.InstallationPath);
            Environment.Exit(0);
        }

        static CheckTarget ParseCommandLineOptionOrDie(string[] args)
        {
            var ret = CheckTarget.BasicBuild;

            foreach (var arg in args)
            {
                switch (arg)
                {
                    case "--basic":
                        ret = CheckTarget.BasicBuild;
                        break;
                    case "--full":
                        ret = CheckTarget.FullDevelopment;
                        break;
                    default:
                        Console.Error.WriteLine($"unknown command line option '{arg}'");
                        Environment.Exit(1);
                        break;
                }
            }

            return ret;
        }

        static bool TestVisualStudioInstance(CheckTarget target, string channelId)
        {
            var vsInstance = GetVisualStudioInstance(target, channelId);
            if (vsInstance == null)
            {
                Console.Error.WriteLine("Visual Studio product not found.");
                return false;
            }

            if (!vsInstance.IsComplete)
            {
                Console.Error.WriteLine($"Visual Studio product '{vsInstance.ProductUniqueId}' found but its installation is incomplete.");
                return false;
            }

            var requiredComponents = GetRequiredComponents(target, channelId);
            var componentSet = new HashSet<string>(vsInstance.Components);
            var missingComponents = requiredComponents.Where(x => !componentSet.Contains(x)).ToArray();
            if (missingComponents.Any())
            {
                Console.Error.WriteLine($"Visual Studio product '{vsInstance.ProductUniqueId}' found but the following required components are missing:");
                foreach (var s in missingComponents)
                {
                    Console.Error.WriteLine(s);
                }
                return false;
            }

            return true;
        }

        static Nact.Utility.VisualStudioInstance GetVisualStudioInstance(CheckTarget target, string channelId)
        {
            switch (target)
            {
                case CheckTarget.BasicBuild:
                    return Nact.Utility.VisualStudioCollection.Instance.GetMsBuild(channelId);
                case CheckTarget.FullDevelopment:
                    return Nact.Utility.VisualStudioCollection.Instance.GetVisualStudio(channelId);
                default:
                    throw new NotImplementedException();
            }
        }

        static IEnumerable<string> GetRequiredComponents(CheckTarget target, string channelId)
        {
            switch (target)
            {
                case CheckTarget.BasicBuild:
                    return ComponentsForBasicDotNetBuild;
                case CheckTarget.FullDevelopment:
                    return ComponentsForFullFeatureBuild;
                default:
                    throw new NotImplementedException();
            }
        }
    }
}
