﻿using Microsoft.VisualStudio.Setup.Configuration;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.Nact.Utility
{
    public class VisualStudioCollection
    {
        public static string VisualStudio15ChannelId { get; } = "VisualStudio.15.Release";
        // VisualStudioCollection.Create には 数 ms かかるので、アクション実行ごとに列挙するには重い。キャッシュしておく。
        // VisualStudioCollection.Create には 数 ms かかるので、アクション実行ごとに列挙するには重い。キャッシュしておく。
        private static VisualStudioCollection s_Instance;
        public static VisualStudioCollection Instance => s_Instance ?? (s_Instance = Create());

        private const int REGDB_E_CLASSNOTREG = unchecked((int)0x80040154);

        private readonly VisualStudioInstance[] m_Instances;


        /// <summary>
        /// Visual Studio 製品の ID。先頭ほど優先度が高い。
        /// </summary>
        private static readonly string[] VisualStudioProductIdsByPrecedence = new string[]
        {
            "Microsoft.VisualStudio.Product.Professional",
            "Microsoft.VisualStudio.Product.Enterprise",
            // 我々は Community のライセンスを受けられないのでサポートしない
            // "Microsoft.VisualStudio.Product.Community",
        };

        /// <summary>
        /// MSBuild を含む製品の ID。先頭ほど優先度が高い。
        /// </summary>
        private static readonly string[] MsBuildProductIdsByPrecedence = new string[]
        {
            "Microsoft.VisualStudio.Product.Professional",
            "Microsoft.VisualStudio.Product.Enterprise",
            // 我々は Community のライセンスを受けられないのでサポートしない
            // "Microsoft.VisualStudio.Product.Community",
            // BuildTools は FxCop などのコンポーネントを含まないので優先度が低い
            "Microsoft.VisualStudio.Product.BuildTools"
        };

        private VisualStudioCollection(IEnumerable<VisualStudioInstance> instances)
        {
            this.m_Instances = instances.ToArray();
        }

        public IReadOnlyCollection<VisualStudioInstance> Instances => m_Instances;

        /// <summary>
        /// 指定のチャンネルにおいて、 Visual Studio 製品を取得します。
        /// </summary>
        /// <returns>Visual Studio 製品。見つからなかった場合 null。</returns>
        public VisualStudioInstance GetVisualStudio(string channelId) => GetProduct(channelId, VisualStudioProductIdsByPrecedence);

        /// <summary>
        /// 指定のチャンネルにおいて、 MSBuild を含む製品を取得します。
        /// </summary>
        /// <returns>Visual Studio 製品。見つからなかった場合 null。</returns>
        public VisualStudioInstance GetMsBuild(string channelId) => GetProduct(channelId, MsBuildProductIdsByPrecedence);

        private VisualStudioInstance GetProduct(string channelId, IEnumerable<string> productIds)
        {
            var instancesFromChannel = m_Instances.Where(x => x.IsFromChannel(channelId)).ToArray();

            foreach (var productId in productIds)
            {
                var instance = instancesFromChannel.FirstOrDefault(x => x.ProductId == productId);
                if (instance != null)
                {
                    return instance;
                }
            }

            return null;
        }

        public static VisualStudioCollection Create()
        {
            return new VisualStudioCollection(EnumerateInstances());
        }

        internal static void ResetCache()
        {
            s_Instance = null;
        }

        private static IEnumerable<VisualStudioInstance> EnumerateInstances()
        {
            IEnumSetupInstances enumerator;

            try
            {
                enumerator = new SetupConfiguration().EnumAllInstances();
            }
            catch (System.Runtime.InteropServices.COMException ex) when (ex.HResult == REGDB_E_CLASSNOTREG)
            {
                // Visual Studio Installer がインストールされていないので、製品も存在しない
                yield break;
            }

            var instance = new ISetupInstance2[1];

            while (true)
            {
                int fetched;
                enumerator.Next(instance.Length, instance, out fetched);

                if (fetched <= 0)
                {
                    yield break;
                }

                yield return new VisualStudioInstance(instance[0]);
            }
        }
    }
}
