﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using System.Linq;

namespace IFFRefGen.Src
{
    // スキーマ関連のグローバルクラス
    public static class GXsd
    {
        #region 初期化
        // スキーマの初期化
        public static void Initialize()
        {
            // スキーマの読み込み
            Read();

            // simpleType の初期化
            for (int i = 0; i < GXsd.SimpleTypes.Count; i++)
            {
                // プリミティブ型を末尾に追加するので for で回す
                GXsd.SimpleTypes[i].Initialize();
            }

            // attributeGroup の初期化
            foreach (XsdAttributeGroup attributeGroup in GXsd.AttributeGroups)
            {
                attributeGroup.Initialize();
                //Debug.WriteLine(attributeGroup);
            }

            string rootElementName = GApp.Config.SchemaRootElement;
            XsdElement rootElement = GXsd.FindElement(rootElementName);
            if (rootElement == null)
            {
                throw new Exception(string.Format("ルート要素が見つかりません {0}",
                    rootElementName));
            }
            rootElement.Initialize();
            // 使用フラグを立てる
            rootElement.Use();
        }

        private static void ParseXmlDocument(string filePath)
        {
            XmlDocument document = new XmlDocument();
            document.Load(filePath);
            foreach (XmlNode node in document.DocumentElement.ChildNodes)
            {
                if (node.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement element = (XmlElement)node;
                string name = element.LocalName;
                if (name == "include")
                {
                    string folderPath = System.IO.Path.GetDirectoryName(filePath);
                    string includeRelativePath = element.GetAttribute("schemaLocation");
                    string includePath = System.IO.Path.Combine(folderPath, includeRelativePath);
                    var includePathUri = new System.Uri(includePath);
                    if (!GXsd.AnalyzedIncludeFiles.Any(uri => includePathUri.Equals(uri)))
                    {
                        ParseXmlDocument(includePath);
                        GXsd.AnalyzedIncludeFiles.Add(includePathUri);
                    }
                }
                else if (name == "simpleType")
                {
                    GXsd.SimpleTypes.Add(new XsdSimpleType(element, filePath));
                }
                else if (name == "attributeGroup")
                {
                    GXsd.AttributeGroups.Add(new XsdAttributeGroup(element, filePath));
                }
                else if (name == "element")
                {
                    GXsd.Elements.Add(new XsdElement(element, filePath));
                }
                else if (name == "complexType")
                {
                    GXsd.ComplexTypes.Add(new XsdComplexType(element, filePath));
                }
                else
                {
                    throw new Exception("スキーマに意図しない要素があります " +
                        name + " " + filePath);
                }
            }
        }

        // スキーマの読み込み
        private static void Read()
        {
            if (string.IsNullOrEmpty(GApp.Config.SchemaFile))
            {
                // スキーマファイルの指定がない場合はフォルダ以下すべての xsd を参照する
                foreach (string filePath in Directory.EnumerateFiles(
                    GApp.Config.SchemaFolder, "*.xsd", SearchOption.AllDirectories))
                {
                    ParseXmlDocument(filePath);
                }
            }
            else
            {
                string schemaFilePath = System.IO.Path.Combine(GApp.Config.SchemaFolder, GApp.Config.SchemaFile);
                ParseXmlDocument(schemaFilePath);
            }
        }
        #endregion

        // 型名が文字列か
        public static bool IsString(string typeName) { return (typeName == GXsd.String); }

        // 型名が数値か
        public static bool IsNumber(string typeName)
        {
            return (
                (typeName == GXsd.Byte) ||
                (typeName == GXsd.UByte) ||
                (typeName == GXsd.Short) ||
                (typeName == GXsd.UShort) ||
                (typeName == GXsd.Int) ||
                (typeName == GXsd.UnsignedInt) ||
                (typeName == GXsd.Float));
        }

        // 型名が日時か
        public static bool IsDateTime(string typeName) { return (typeName == GXsd.DateTime); }

        // element の検索
        public static XsdElement FindElement(string name)
        {
            return GXsd.Elements.Find(
                delegate(XsdElement match) { return (match.Name == name); });
        }

        // 参照 element の検索
        public static List<XsdElement> FindReferenceElements(XsdElement target)
        {
            List<XsdElement> result = new List<XsdElement>();
            foreach (XsdElement element in GXsd.Elements)
            {
                XsdComplexType type = element.Type as XsdComplexType;
                if (type == null) { continue; }

                FindReferenceElementsContent(result, target.Name, element, type.Content);
            }
            return result;
        }

        // 参照 element の content 検索
        private static void FindReferenceElementsContent(
            List<XsdElement> result,
            string targetName,
            XsdElement element,
            XsdContent content)
        {
            if (content == null) { return; }

            XsdContentAny any = content as XsdContentAny;
            if (any != null) { return; }

            XsdContentReference reference = content as XsdContentReference;
            if (reference != null)
            {
                if (reference.ReferenceName == targetName) { result.Add(element); }
                return;
            }

            XsdContentSequence sequence = content as XsdContentSequence;
            if (sequence != null)
            {
                foreach (XsdContent child in sequence.Contents)
                {
                    FindReferenceElementsContent(result, targetName, element, child);
                }
                return;
            }

            XsdContentChoice choice = content as XsdContentChoice;
            if (choice != null)
            {
                foreach (XsdContent child in choice.Contents)
                {
                    FindReferenceElementsContent(result, targetName, element, child);
                }
            }
        }

        // complexType の検索
        public static XsdComplexType FindComplexType(string name)
        {
            return GXsd.ComplexTypes.Find(
                delegate(XsdComplexType match) { return (match.Name == name); });
        }

        // simpleType の検索
        public static XsdSimpleType FindSimpleType(string name)
        {
            return GXsd.SimpleTypes.Find(
                delegate(XsdSimpleType match) { return (match.Name == name); });
        }

        // attributeGroup の検索
        public static XsdAttributeGroup FindAttributeGroup(string name)
        {
            return GXsd.AttributeGroups.Find(
                delegate(XsdAttributeGroup match) { return (match.Name == name); });
        }

        /// TODO: Dictionary 化
        // element
        public static readonly List<XsdElement> Elements = new List<XsdElement>();
        // comlexType
        public static readonly List<XsdComplexType> ComplexTypes = new List<XsdComplexType>();
        // simpleType
        public static readonly List<XsdSimpleType> SimpleTypes = new List<XsdSimpleType>();
        // attributeGroup
        public static readonly List<XsdAttributeGroup> AttributeGroups = new List<XsdAttributeGroup>();

        public static readonly List<Uri> AnalyzedIncludeFiles = new List<Uri>();

        public const string Namespace = "xsd:";
        public const string Boolean = Namespace + "boolean";
        public const string Byte = Namespace + "byte";
        public const string UByte = Namespace + "unsignedByte";
        public const string Short = Namespace + "short";
        public const string UShort = Namespace + "unsignedShort";
        public const string Int = Namespace + "int";
        public const string UnsignedInt = Namespace + "unsignedInt";
        public const string Float = Namespace + "float";
        public const string HexBinary = Namespace + "hexBinary";
        public const string String = Namespace + "string";
        public const string DateTime = Namespace + "dateTime";
    }
}
