﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Collections;
using System.IO;
using System.Xml;
using System.Text.RegularExpressions;
using System.IO.Compression;
using System.Globalization;

using LOG = Common.Logger;
using FILEIO = Common.FileIOWrapper;

namespace HtmlToVsHelp
{
    public partial class VsHelpConvert
    {
        const string VsHelpIdFormat = "{0}-{1:X8}-{2:X6}";
        const string VsHelpUriTop = "ms-xhelp:///?Id=";
        const string DefaultTitleName = @"NintendoSDK API Reference";
        const string DefaultDocumentName = @"index";

        private struct TocInfo
        {
            public string TocParent;
            public int TocOrder;

            public TocInfo(string TocParent, int TocOrder)
            {
                this.TocParent = TocParent;
                this.TocOrder = TocOrder;
            }
        }
        private static Dictionary<string, string> idtable = new Dictionary<string, string>();
        private static Dictionary<string, TocInfo> tocInfoTable = new Dictionary<string, TocInfo>();
        private static Dictionary<string, int> fileIndexTable = new Dictionary<string, int>();

        private static string doxygenInputPath = null;
        private static string doxygenOutputPath = null;

        private static string titleName = DefaultTitleName;
        private static string baseIdName = DefaultDocumentName;

        /// <summary>
        /// ドキュメントパスを指定します
        /// </summary>
        /// <param name="doxygenpath">ドキュメントパス(%SIGLO_ROOT%\Documents\Outputs\Api 相当)</param>
        /// <returns>パスが存在して設定できたら true</returns>
        public static bool SetPath(string doxygenPath)
        {
            if (!Directory.Exists(doxygenPath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "document input path is not found ({0})\n", doxygenPath);
                return false;
            }

            Common.UtilFs.SetVsHelpBaseDirName(CmdlineOption.InputDirectoryName);
            Common.UtilFs.SetVsHelpDirName(CmdlineOption.OutputDirectoryName);
            doxygenInputPath = doxygenPath + Common.UtilFs.GetHtmlDirName();
            doxygenOutputPath = doxygenPath + Common.UtilFs.GetHelpOutputDirName();

            // doxygen HTMLファイル存在チェック用ハッシュの初期化
            Common.UtilFs.InitalizeDoxygenHtmlFileNameHash(doxygenInputPath);

            return true;
        }

        public static bool SetTitle(string title)
        {
            if (title == null)
            {
                return true;
            }

            titleName = title;
            return true;
        }

        public static bool SetDocumentName(string name)
        {
            if (name == null)
            {
                return true;
            }

            baseIdName = name;
            return true;
        }

        /// <summary>
        /// VsHelp形式ヘルプへの変換前処理を行います
        /// </summary>
        public static void PreConvert()
        {
            if (doxygenInputPath == null)
            {
                return;
            }

            //IDテーブルの初期化
            InitializeIdTable();

            // コンテンツの登録
            //  (コンテンツ階層情報を登録します)
            if (RegisterContentsInformation() == false)
            {
                LOG.LogLine(LOG.Level.LOG_ERROR, "Register content information failed.");
                return;
            }

            // コンテンツ登録情報の確認
            CheckInfomationTables();
        }

        private static void InitializeIdTable()
        {
            // Microsoft.Help.Idのベース名を作成
            //  "index-(16進32bit値)-XXXXXX"という値
            // index.htmlはXXXXXを000000とする
            string[] files = Common.UtilFs.GetDoxygenHtmlFiles();
            string filename;
            int fileCount = 0;
            string idname = null;

            // Idの設定
            int seed = Environment.TickCount;
            Random rnd = new Random(seed);
            int indexNumber = rnd.Next(1, 0x7FFFFFFF);

            foreach (string f in files)
            {
                filename = f;
                if (filename == "index.html")
                {
                    idname = string.Format(VsHelpIdFormat, baseIdName, indexNumber, 0);
                    idtable[filename] = idname;
                    continue;
                }
                fileCount++;
                idname = string.Format(VsHelpIdFormat, baseIdName, indexNumber, fileCount);
                idtable[filename] = idname;
            }
        }

        private static bool RegisterContentsInformation()
        {
#if true
            // TocParent・TocOrderの設定
            //   TocPartBranchTree を使ったパート別階層登録を行っている

            if (SetTocInformation() == false)
            {
                return false;
            }

            return ConvertSocInfoToTable();
#else
            // TocParent・TocOrderの設定
            //   目次階層の関連付けはここで行う
            //   親ページ・子ページともファイル名が固定の場合は SetVsHelpPageParent() を呼び出す
            //   子ページが正規表現でまとめられる場合は SetVsHelpPageParent() を呼び出す
            //   子ページ・親ページ共に正規表現を適用する必要がある場合は SetVsHelpPageParentToNamespace() を呼び出す

            // 現在この方式は使用していないが、使用は可能である

            return true;
#endif
        }

        private static bool CheckInfomationTables()
        {
            // 漏れチェック
            foreach (string file in idtable.Keys)
            {
                if (!tocInfoTable.ContainsKey(file))
                {
                    LOG.LogLine(LOG.Level.LOG_WARN, "file \"{0}\" is not registered to VsHelp content list.", file);
                }
            }
            foreach (string file in tocInfoTable.Keys)
            {
                if (!idtable.ContainsKey(file))
                {
                    LOG.LogLine(LOG.Level.LOG_ERROR, "file ID not found : \"{0}\"", file);
                }
            }
            return true;
        }

        private static bool IsRegexString(string str)
        {
            // ルールは [ のみ。
            return str.Contains("[");
        }

        /// <summary>
        /// ヘルプファイルの目次階層を一括登録する
        /// </summary>
        /// <param name="topParentFile">一括登録するファイル群の一番親のファイル名</param>
        /// <param name="fixedNamespaceString">ファイルの親がルール1・ルール2に該当する場合にこの引数で接頭辞を指定できる</param>
        /// <param name="targetFileRuleString">一括登録するファイル群の正規表現</param>
        /// <param name="rule1String">ルール1の正規表現を指定する。一括登録するファイル群のうちこのルールに適用されるものは第1グループがファイル名となる</param>
        /// <param name="rule2String">ルール2の正規表現を指定する。一括登録するファイル群のうちこのルールに適用されるものは第1グループがファイル名となる</param>
        private static void SetVsHelpPageParentToNamespace(string topParentFileName, string fixedNamespaceString, string targetFileRuleString, string rule1String, string rule2String = null)
        {
            bool isTopParentfile = (topParentFileName != null && topParentFileName != string.Empty);
            string fixedString = fixedNamespaceString != null ? fixedNamespaceString : string.Empty;
            Regex regex = new Regex(targetFileRuleString);
            foreach (string tfile in from file in Common.UtilFs.GetDoxygenHtmlFiles() where regex.IsMatch(file) select Path.GetFileName(file))
            {
                bool isRule2 = (rule2String != null && rule2String != string.Empty);
                Regex regrex2 = new Regex(rule1String);
                Match m2 = regrex2.Match(tfile);
                if (m2.Success)
                {
                    // クラスメンバのページは所属する名前空間のページの下にする
                    string parentFile = fixedString + m2.Groups[1].Value + ".html";
                    if (Common.UtilFs.CheckDoxygenHtmlFileExist(parentFile))
                    {
                        SetVsHelpPageParentCore(tfile, parentFile);
                        continue;
                    }
                    else if (CmdlineOption.CheckParentPage == true)
                    {
                        LOG.LogLine(LOG.Level.LOG_WARN, "parent page is not found (child={0}, parent={1})", tfile, parentFile);
                    }
                }
                if (isRule2 == true)
                {
                    Regex regrex3 = new Regex(rule2String);
                    Match m3 = regrex3.Match(tfile);
                    if (m3.Success)
                    {
                        // メンバ一覧のページは所属する名前空間のページの下にする
                        string parentFile = fixedString + m3.Groups[1].Value + ".html";
                        if (Common.UtilFs.CheckDoxygenHtmlFileExist(parentFile))
                        {
                            SetVsHelpPageParentCore(tfile, parentFile);
                            continue;
                        }
                        else if (CmdlineOption.CheckParentPage == true)
                        {
                            LOG.LogLine(LOG.Level.LOG_WARN, "parent page is not found (child={0}, parent={1})", tfile, parentFile);
                        }
                    }
                }
                if (isTopParentfile == true)
                {
                    SetVsHelpPageParentCore(tfile, topParentFileName);
                }
            }
        }

        /// <summary>
        /// ヘルプファイルの目次階層を登録する
        /// </summary>
        /// <param name="fileNameRegex">登録するファイルの正規表現</param>
        /// <param name="parentFileName">親階層のファイル名</param>
        private static void SetVsHelpPageParent(string fileNameRegex, string parentFileName)
        {
#if false
            if (!Common.UtilFs.CheckDoxygenHtmlFileExist(parentFileName))
            {
                LOG.LogLine(LOG.Level.LOG_WARN, @"SetVsHelpPageParent : Parent file is not found. ({0})", parentFileName);
                return;
            }
#endif

            if (IsRegexString(fileNameRegex))
            {
                Regex regex = new Regex(fileNameRegex);
                foreach (string tfile in from file in Common.UtilFs.GetDoxygenHtmlFiles() where regex.IsMatch(file) select Path.GetFileName(file))
                {
                    SetVsHelpPageParentCore(tfile, parentFileName);
                }
            }
            else
            {
                SetVsHelpPageParentCore(fileNameRegex, parentFileName);
            }
        }

        /// <summary>
        /// ヘルプファイルの目次階層を登録する(parentFileName がない場合は grandParentFileName を使う)
        /// </summary>
        /// <param name="fileNameRegex">登録するファイルの正規表現</param>
        /// <param name="parentFileName">親階層のファイル名</param>
        /// <param name="elseParentFileName">parentFileName がない時のファイル名</param>
        private static void SetVsHelpPageParentWithElse(string fileNameRegex, string parentFileName, string elseParentFileName)
        {
            string parentfile = parentFileName;
            if (!Common.UtilFs.CheckDoxygenHtmlFileExist(parentFileName))
            {
                parentfile = elseParentFileName;
            }
            else if (!Common.UtilFs.CheckDoxygenHtmlFileExist(elseParentFileName))
            {
                LOG.LogLine(LOG.Level.LOG_WARN, @"SetVsHelpPageParent : Both parent files are not found. ({0}, {1})", parentFileName, elseParentFileName);
                return;
            }

            if (IsRegexString(fileNameRegex))
            {
                Regex regex = new Regex(fileNameRegex);
                foreach (string tfile in from file in Common.UtilFs.GetDoxygenHtmlFiles() where regex.IsMatch(file) select Path.GetFileName(file))
                {
                    SetVsHelpPageParentCore(tfile, parentfile);
                }
            }
            else
            {
                SetVsHelpPageParentCore(fileNameRegex, parentfile);
            }
        }

        private static void SetVsHelpPageParentCore(string fileName, string parentFileName)
        {
            if (fileName != "index.html")
            {
                if (fileName == parentFileName)
                {
                    LOG.LogLine(LOG.Level.LOG_WARN, "SetVsHelpPageParentCore : parentFileName must not same with fileName({0}).", fileName);
                    return;
                }
                if (!Common.UtilFs.CheckDoxygenHtmlFileExist(fileName))
                {
                    // 実在しないファイルの場合は登録しない
                    return;
                }
                if (!idtable.ContainsKey(parentFileName))
                {
                    // CheckInfomationTables() で一括チェックを行うためここではログは出さない
                    return;
                }
                int index;
                // 親ページの重複数を確認しインクリメントする
                if (fileIndexTable.TryGetValue(parentFileName, out index))
                {
                    index++;
                }
                else
                {
                    index = 0;
                }
                fileIndexTable[parentFileName] = index;
                // 親ページのIDと現在の重複数を登録する
                tocInfoTable[fileName] = new TocInfo(idtable[parentFileName], index);
            }
            else
            {
                // 自分が親なので登録する
                tocInfoTable[fileName] = new TocInfo("-1", 0);
            }
        }

        private enum ParseMode
        {
            PARSE_HEADER1,
            PARSE_HEADER2,
            PARSE_PRE,
            PARSE_OTHER
        }

        /// <summary>
        /// doxygen生成HTMLファイルをVsHelpで利用できるように変換を行います
        /// </summary>
        public static void Convert()
        {
            if (doxygenInputPath == null)
            {
                return;
            }

            string[] filenames = Common.UtilFs.GetDoxygenHtmlFiles();

            // HTMLファイルをVsHelp用ファイルに変換する
            CreateAllConvertFiles(filenames);
        }

        private static bool CreateAllConvertFiles(string[] filenames)
        {
            foreach (string f in filenames)
            {
                string filename = f;

                // ヘルプ作成の想定外ファイルはスキップする
                // MEMO: 既に存在チェックしているためここではエラーログは出力しない
                if (!tocInfoTable.ContainsKey(filename))
                {
                    continue;
                }

                // 変換の実処理
                CreateSeveralConvertFile(filename);
            }

            return true;
        }

        private static bool CreateSeveralConvertFile(string filename)
        {
            string filepath = doxygenInputPath + Path.DirectorySeparatorChar + filename;
            Encoding encode = Encoding.GetEncoding("UTF-8");

            // ファイルを開く
            string fileData;
            FILEIO.GetData(out fileData, filepath);
            using (StringReader sr = new StringReader(fileData))
            using (StringWriter sw = new StringWriter())
            {
                sw.NewLine = "\n";

                ParseMode mode = ParseMode.PARSE_HEADER1;

                sw.WriteLine("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
                sw.WriteLine("<html xmlns:MSHelp=\"http://msdn.microsoft.com/mshelp\" xmlns:mshelp=\"http://msdn.microsoft.com/mshelp\" xmlns=\"http://www.w3.org/1999/xhtml\">");

                // <head>までは何もしない
                // </head>が見つかったらその直前にmetaヘッダを追記
                //  Microsoft.Help.TocParentはindex.htmlのものにする
                // <a href="...">を変換する (ただし <pre>～</pre> の行はスキップする)
                string line;
                while (sr.Peek() > -1)
                {
                    line = sr.ReadLine();
                    if (line.StartsWith("<head>"))
                    {
                        mode = ParseMode.PARSE_HEADER2;
                    }
                    else if (line.Contains("<pre ") || line.Contains("<pre>"))
                    {
                        mode = ParseMode.PARSE_PRE;
                    }

                    if (line.Contains("href="))
                    {
                        if (mode != ParseMode.PARSE_HEADER2 && mode != ParseMode.PARSE_PRE)
                        {
                            string newline = ConvertLinkUri(line);
                            sw.WriteLine(newline);

                            mode = ParseMode.PARSE_OTHER;
                            continue;
                        }
                    }

                    if (line.StartsWith("</head>"))
                    {
                        //metaを書き込む
                        WriteMetaHeaders(sw, filename);

                        mode = ParseMode.PARSE_OTHER;
                    }
                    else if (line.Contains("</pre>"))
                    {
                        mode = ParseMode.PARSE_OTHER;
                    }

                    switch (mode)
                    {
                        case ParseMode.PARSE_HEADER1:
                            //何もしない
                            break;
                        case ParseMode.PARSE_HEADER2:
                        case ParseMode.PARSE_PRE:
                        case ParseMode.PARSE_OTHER:
                            //そのまま書き込む
                            sw.WriteLine(line);
                            break;
                        default:
                            LOG.LogLine(LOG.Level.LOG_ERROR, "ParseMode error {0}", mode);
                            break;
                    }
                }

                string data = sw.ToString();
                FILEIO.SetData(ref data, filepath);
            }

            return true;
        }

        private static bool WriteMetaHeaders(StringWriter sw, string filename)
        {
            //metaを書き込む
            TocInfo info = tocInfoTable[filename];
            switch (CmdlineOption.CountryMode)
            {
                case CmdlineOption.LanguageModeType.LANGUAGE_EN_US:
                    sw.WriteLine("<meta name=\"Microsoft.Help.Locale\" content=\"en-us\" />");
                    sw.WriteLine("<meta name=\"Microsoft.Help.TopicLocale\" content=\"en-us\" />");
                    break;
                default:
                    sw.WriteLine("<meta name=\"Microsoft.Help.Locale\" content=\"ja-jp\" />");
                    sw.WriteLine("<meta name=\"Microsoft.Help.TopicLocale\" content=\"ja-jp\" />");
                    break;
            }
            sw.WriteLine("<meta name=\"Microsoft.Help.SelfBranded\" content=\"true\" />");
            sw.WriteLine("<meta name=\"Microsoft.Help.Id\" content=\"{0}\" />", idtable[filename]);
            sw.WriteLine("<meta name=\"Microsoft.Help.TocParent\" content=\"{0}\" />", info.TocParent);
            sw.WriteLine("<meta name=\"Microsoft.Help.TocOrder\" content=\"{0}\" />", info.TocOrder);
            sw.WriteLine("<meta name=\"Microsoft.Help.Category\" content=\"" + baseIdName + "\" />");
            sw.WriteLine("<meta name=\"Microsoft.Help.DisplayVersion\" content=\"\" />");

            return true;
        }

        private static string ConvertLinkUri(string line)
        {
            //ページへのリンクをVsHelp形式のIDに変換する
            Regex reg = new Regex("<(a|link)[^>]* href=\"(?<url>.*?)\"[^>]*>", RegexOptions.IgnoreCase | RegexOptions.Singleline);
            string newline = line;
            Match m = reg.Match(line);
            do
            {
                if (m.Success == false)
                {
                    LOG.LogLine(LOG.Level.LOG_ERROR, "href not found ({0})", line);
                    break;
                }
                string url = m.Groups["url"].Value;
                if (!url.StartsWith("http://"))
                {
                    string urlfile = DeleteUrlHash(url);
                    if (urlfile.EndsWith(".html") || urlfile.EndsWith(".HTML"))
                    {
                        string msurl = ConvertHelpPathToId(urlfile);
                        // MEMO: 複数存在する場合はまとめて変換される
                        newline = newline.Replace(urlfile, msurl);
                    }
                }

                m = m.NextMatch();
            } while (m.Success);

            return newline;
        }

        private static string DeleteUrlHash(string url)
        {
            if (url.Contains('#'))
            {
                return url.Substring(0, url.IndexOf("#"));
            }
            else
            {
                return url;
            }
        }

        private static string ConvertHelpPathToId(string urlfile)
        {
            string idtablefile = null;
            if (idtable.TryGetValue(urlfile, out idtablefile))
            {
                return VsHelpUriTop + idtablefile;
            }
            else
            {
                return VsHelpUriTop + idtable["index.html"];
            }
        }

        /// <summary>
        /// doxygen生成HTMLファイル内の特殊文字をVsHelpとして表示できるように変換します
        /// </summary>
        public static void ConvertHtmlCharacter()
        {
            if (doxygenInputPath == null)
            {
                return;
            }
            string[] files = Common.UtilFs.GetDoxygenHtmlFiles();
            string filepath;
            string filename;
            Encoding encode = Encoding.GetEncoding("UTF-8");
            string line;

            foreach (string f in files)
            {
                filename = f;
                filepath = doxygenInputPath + Path.DirectorySeparatorChar + f;

                // ヘルプ作成の想定外ファイルはスキップする
                if (!tocInfoTable.ContainsKey(filename))
                {
                    continue;
                }

                // ファイルを開く
                string fileData;
                FILEIO.GetData(out fileData, filepath);
                using (StringReader sr = new StringReader(fileData))
                using (StringWriter sw = new StringWriter())
                {
                    sw.NewLine = "\n";
                    int lineNum = 0;

                    // 特殊文字を見つけたら置き換えて書き込む
                    // MEMO: "&nbsp;"についてはHtmlViewer側で不具合があるため半角空白にする
                    // さらにタイトルをコンテンツ一覧で見やすくなるように変換する
                    while (sr.Peek() > -1)
                    {
                        lineNum++;
                        line = sr.ReadLine();
                        line = ConvertSpecialCharacters(line);
                        line = ConvertUnCommentedSpecialCharacters(line);
                        line = ChangeTitleName(filename, line);
#if DEBUG
                        CheckSpecialCharacter(line, filename, lineNum);
#endif
                        sw.WriteLine(line);
                    }

                    string data = sw.ToString();
                    FILEIO.SetData(ref data, filepath);
                }
            }
        }

        private static Dictionary<string, string> specialCharacterDef = new Dictionary<string, string>()
                                                                            { { "&nbsp;", " " },
                                                                              { "&copy;", "&#169;" },
                                                                              { "&quot;", "&#34;" },
                                                                              { "&amp;", "&#38;" },
                                                                              { "&lt;", "&#60;" },
                                                                              { "&gt;", "&#62;" },
                                                                              { "&mdash;", "&#8212;" },
                                                                              { "&ndash;", "&#8211;" }
                                                                            };

        private static string ConvertSpecialCharacters(string line)
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder(line);
            foreach (KeyValuePair<string, string> def in specialCharacterDef)
            {
                sb.Replace(def.Key, def.Value);
            }
            return sb.ToString();
        }

        private enum SpecialCharacterParseMode
        {
            charMode_Tag,
            charMode_Other
        }

        private static string ConvertUnCommentedSpecialCharacters(string line)
        {
#if false
            // 特殊文字がない場合は変換を省略する
            string[] check = {"\"", "&", "<", ">", "©"};
            bool charCheck = false;
            foreach (string chk in check)
            {
                if (line.Contains(chk))
                {
                    charCheck = true;
                }
            }
            if (!charCheck)
            {
                return line;
            }
#endif
            SpecialCharacterParseMode mode = SpecialCharacterParseMode.charMode_Other;
            TextElementEnumerator charEnum = StringInfo.GetTextElementEnumerator(line);
            StringBuilder analyzedString = new StringBuilder();
            bool isCheckTag = false;
            bool isQuote = false;
            while (charEnum.MoveNext())
            {
                if (isCheckTag == true)
                {
                    // MEMO: '<'の判定について以下の場合にタグ先頭と判定する
                    //         ・次の文字が半角アルファベット・“!”・“/”・“?”のいずれか
                    //         ・次の文字がアルファベットでその次に来る空白の直前の文字が“,”ではないの場合
                    //         ・次の文字がアルファベットでタグ内に空白文字および“.”が共にない場合
                    //       それ以外の場合は対応しない
                    //       また、'<' が行の終端だった場合は変換できない
                    Encoding encode = Encoding.GetEncoding("UTF-8");
                    string current = charEnum.GetTextElement();
                    if ((encode.GetByteCount(current) > 1 || !char.IsLetter(current[0])) && current != "!" && current != "/" && current != "?")
                    {
                        mode = SpecialCharacterParseMode.charMode_Other;
                        analyzedString.Append(@"&#61;");
                    }
                    else if (encode.GetByteCount(current) == 1 && char.IsLetter(current[0]))
                    {
                        // ' ' '>' までで切り取って残りの文字列の終端が ',' の場合はタグではない
                        // ' ' がなく '.' もない場合はタグではない
                        int pos = charEnum.ElementIndex;
                        string checkstring = line.Substring(pos, line.Length - pos);
                        if (checkstring.Contains('>'))
                        {
                            checkstring = checkstring.Substring(0, checkstring.IndexOf('>') + 1);
                        }

                        if (checkstring.Contains(' '))
                        {
                            if (checkstring.IndexOf(' ') >= 0)
                            {
                                checkstring = checkstring.Substring(0, checkstring.IndexOf(' '));
                            }
                            if (!checkstring.Contains(','))
                            {
                                analyzedString.Append("<");
                            }
                            else
                            {
                                mode = SpecialCharacterParseMode.charMode_Other;
                                analyzedString.Append(@"&#61;");
                            }
                        }
                        else
                        {
                            if (!checkstring.Contains('.'))
                            {
                                analyzedString.Append("<");
                            }
                            else
                            {
                                mode = SpecialCharacterParseMode.charMode_Other;
                                analyzedString.Append(@"&#61;");
                            }
                        }
                    }
                    else
                    {
                        analyzedString.Append("<");
                    }
                    isCheckTag = false;
                }

                switch (charEnum.Current.ToString())
                {
                    case "<":
                        if (isQuote == true)
                        {
                            analyzedString.Append(@"&#61;");
                        }
                        else
                        {
                            mode = SpecialCharacterParseMode.charMode_Tag;
                            isCheckTag = true;
                        }
                        break;
                    case ">":
                        if (isQuote == true)
                        {
                            analyzedString.Append(@"&#62;");
                        }
                        else if (mode == SpecialCharacterParseMode.charMode_Tag)
                        {
                            mode = SpecialCharacterParseMode.charMode_Other;
                            analyzedString.Append(charEnum.Current);
                        }
                        else
                        {
                            // 変換する
                            analyzedString.Append(@"&#62;");
                        }
                        break;
                    case "&":
                        // 特殊文字が続く場合はスルーして文字の位置を増分する(エラーチェックすること)
                        int specialLength = CheckEscapeString(ref charEnum, line);
                        if (specialLength > 0)
                        {
                            analyzedString.Append(charEnum.Current);
                        }
                        for (int count = 0; count < specialLength - 1; count++)
                        {
                            if (charEnum.MoveNext() == false)
                            {
                                break;
                            }
                            analyzedString.Append(charEnum.Current);
                        }
                        // 特殊文字が続かない場合は変換する
                        if (specialLength == 0)
                        {
                            analyzedString.Append(@"&#38;");
                        }
                        break;
                    case "\"":
                        // otherモードの場合は変換する
                        if (mode == SpecialCharacterParseMode.charMode_Other)
                        {
                            analyzedString.Append(@"&#34;");
                        }
                        else
                        {
                            if (isQuote == true)
                            {
                                isQuote = false;
                            }
                            else
                            {
                                isQuote = true;
                            }
                            analyzedString.Append(charEnum.Current);
                        }
                        break;
                    case "©":
                        // otherモードの場合は変換する
                        if (mode == SpecialCharacterParseMode.charMode_Other)
                        {
                            analyzedString.Append(@"&#169;");
                        }
                        else
                        {
                            analyzedString.Append(charEnum.Current);
                        }
                        break;
                    case "'":
                        // otherモードの場合は変換する
                        if (mode == SpecialCharacterParseMode.charMode_Other)
                        {
                            analyzedString.Append(@"&#8217;");
                        }
                        else
                        {
                            analyzedString.Append(charEnum.Current);
                        }
                        break;
                    default:
                        // そのまま登録する
                        analyzedString.Append(charEnum.Current);
                        break;
                }
            }

            return analyzedString.ToString();
        }
        private static int CheckEscapeString(ref TextElementEnumerator charEnum, string line)
        {
            int index = charEnum.ElementIndex;
            int maxlength = line.Length - index;

            // System.Diagnostics.Debug.Write(string.Format("sub {0}, {1}, {2}\n", index, maxlength, line.Length));
            string checkString = line.Substring(index, maxlength < 12 ? maxlength : 12);
            Regex regex = new Regex(@"^&[a-zA-Z]+;");
            Match match = regex.Match(checkString);
            if (match.Success)
            {
                return match.Length;
            }
            Regex regex2 = new Regex(@"^&#[0-9]+;");
            match = regex2.Match(checkString);
            if (match.Success)
            {
                return match.Length;
            }
            return 0;
        }

        private enum SpecialCharacterCheckMode
        {
            SpecialCharacterCheck_Amp,
            SpecialCharacterCheck_Character,
            SpecialCharacterCheck_None
        }

        private static bool CheckSpecialCharacter(string line, string fileName, int lineNumber)
        {
            //TODO: 今は特殊文字が行で一つのみ探せる。複数対応した方がいい
            SpecialCharacterCheckMode mode = SpecialCharacterCheckMode.SpecialCharacterCheck_None;
            char[] lineChars = line.ToCharArray();
            StringBuilder spcialString = new StringBuilder();
            foreach (char current in lineChars)
            {
                if (current == '&')
                {
                    mode = SpecialCharacterCheckMode.SpecialCharacterCheck_Amp;
                    spcialString.Append(current);
                }
                else if (current == '#')
                {
                    if (mode == SpecialCharacterCheckMode.SpecialCharacterCheck_Amp)
                    {
                        mode = SpecialCharacterCheckMode.SpecialCharacterCheck_None;
                    }
                }
                else if (IsCharacter(current))
                {
                    if (mode == SpecialCharacterCheckMode.SpecialCharacterCheck_Amp)
                    {
                        mode = SpecialCharacterCheckMode.SpecialCharacterCheck_Character;
                        spcialString.Append(current);
                    }
                    else if (mode == SpecialCharacterCheckMode.SpecialCharacterCheck_Character)
                    {
                        spcialString.Append(current);
                    }
                }
                else if (current == ';')
                {
                    if (mode == SpecialCharacterCheckMode.SpecialCharacterCheck_Character)
                    {
                        spcialString.Append(current);
                        LOG.LogLine(LOG.Level.LOG_WARN, @"special character ""{0}"" found : {1}::{2}", spcialString.ToString(), fileName, lineNumber);
                        return false;
                    }
                }
                else
                {
                    if (mode != SpecialCharacterCheckMode.SpecialCharacterCheck_None)
                    {
                        mode = SpecialCharacterCheckMode.SpecialCharacterCheck_None;
                    }
                }
            }
            return true;
        }

        private static bool IsCharacter(char c)
        {
            if ('0' <= c && c <= '9')
            {
                return true;
            }
            else if ('a' <= c && c <= 'z')
            {
                return true;
            }
            else if ('A' <= c && c <= 'Z')
            {
                return true;
            }
            return false;
        }

        private static string ChangeTitleName(string filename, string line)
        {
            if (line.Contains("<title>"))
            {
                if (filename == "index.html")
                {
                    return "<title>" + titleName + "</title>";
                }
                else
                {
                    return line.Replace(titleName + ": ", string.Empty);
                }
            }
            return line;
        }

        private static string GetHelpIdBaseName(string title)
        {
            string returnString = null;
            returnString = title.Replace(' ', '_').Replace('"', '_').Replace('\'', '_');
            return returnString;
        }

        /// <summary>
        /// 変換処理を行ったHTMLファイルをVsHelpヘルプファイルに圧縮します
        /// </summary>
        public static void MakeHelpArchive()
        {
            if (doxygenInputPath == null || doxygenOutputPath == null)
            {
                return;
            }
            if (!Directory.Exists(doxygenOutputPath))
            {
                Directory.CreateDirectory(doxygenOutputPath);
            }
            string mshaFilePath = doxygenOutputPath + "\\helpcontentsetup.msha";
            Common.UtilFs.FileCheckDelete(mshaFilePath);
            byte[] resourceBytes;
            switch (CmdlineOption.CountryMode)
            {
                case CmdlineOption.LanguageModeType.LANGUAGE_EN_US:
                    resourceBytes = Properties.Resources.helpcontentsetup_us;
                    break;
                default:
                    resourceBytes = Properties.Resources.helpcontentsetup;
                    break;
            }
            string text = Encoding.Unicode.GetString(resourceBytes);
            text = text.Replace("{0}", CmdlineOption.FlagPrepareCabinet ? baseIdName + ".cab" : baseIdName + ".mshc");
            text = text.Replace("{1}", titleName);
            text = text.Replace("{2}", baseIdName);

            File.WriteAllBytes(mshaFilePath, Encoding.Unicode.GetBytes(text));

            string mshcFilePath = doxygenOutputPath + "\\" + baseIdName + ".mshc";
            CompressionLevel compressLevel = CmdlineOption.FlagPrepareCabinet ? CompressionLevel.NoCompression : CompressionLevel.Optimal;
            Common.UtilFs.FileCheckDelete(mshcFilePath);
            ZipFile.CreateFromDirectory(doxygenInputPath, mshcFilePath, compressLevel, false);
        }

        public static void MakeLstFile()
        {
            if (doxygenOutputPath == null || !CmdlineOption.FlagPrepareCabinet)
            {
                return;
            }
            if (!Directory.Exists(doxygenOutputPath))
            {
                Directory.CreateDirectory(doxygenOutputPath);
            }
            string lstFilePath = doxygenOutputPath + "\\cabinet.lst";

            using (StringWriter sw = new StringWriter())
            {
                sw.NewLine = "\n";

                sw.WriteLine(@".Set MaxDiskSize=0");
                sw.WriteLine(@".Set CabinetNameTemplate=" + baseIdName + @".cab");
                sw.WriteLine(@".Set DiskDirectoryTemplate=""" + doxygenOutputPath + @"""");
                sw.WriteLine(@".Set RptFileName=NUL");
                sw.WriteLine(@".Set InfFileName=NUL");
                sw.WriteLine(@"""" + doxygenOutputPath + @"\" + baseIdName + @".mshc""");
                sw.WriteLine(@"""" + doxygenOutputPath + @"\" + baseIdName + @".mshi""");

                File.WriteAllText(lstFilePath, sw.ToString());
            }

        }
    }
}
