﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Globalization;

using LOG = Common.Logger;

namespace Common
{
    public class ParseDoxygenConf
    {
        private static string doxygenConfigPath = null;

        private enum DoxygenConfParseMode
        {
            DOXYGEN_PARSE_INPUT,
            DOXYGEN_PARSE_OTHER
        }

        public static void SetPath(string configPath)
        {
            if (!File.Exists(configPath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "file not found ({0})\n", configPath);

                return;
            }
            doxygenConfigPath = configPath;
        }

        public static string[] GetValue(string key)
        {
            List<string> list = new List<string>();
            if (!File.Exists(doxygenConfigPath))
            {
                return new string[0];
            }

            Encoding encode = Encoding.GetEncoding("UTF-8");
            using (StreamReader sr = new StreamReader(doxygenConfigPath, encode))
            {
                string line;
                DoxygenConfParseMode mode = DoxygenConfParseMode.DOXYGEN_PARSE_OTHER;

                while (sr.Peek() > -1)
                {
                    line = sr.ReadLine();

                    switch (mode)
                    {
                        case DoxygenConfParseMode.DOXYGEN_PARSE_OTHER:
                            //定義1行目の解析
                            if (CheckFirstLine(line))
                            {
                                mode = ParseFirstLine(mode, line, key, ref list);
                            }
                            break;
                        case DoxygenConfParseMode.DOXYGEN_PARSE_INPUT:
                            //定義2行目以降の解析
                            {
                                mode = ParseSecondLine(mode, line, key, ref list);
                            }
                            break;
                    }
                }
            }

            return list.ToArray();
        }

        private static DoxygenConfParseMode ParseFirstLine(DoxygenConfParseMode mode, string line, string key, ref List<string> list)
        {
            // keyと一致するか確認
            // 値の切り出し
            // 次行確認
            DoxygenConfParseMode retmode = mode;
            string linekey = GetKeyName(line);
            if (linekey == key)
            {
                bool clear;
                int valueStart = GetValueStart(out clear, line);
                if (valueStart >= 0)
                {
                    if (clear)
                    {
                        list.Clear();
                    }
                    string[] value = GetValueName(line, valueStart);
                    if (value.Length > 0)
                    {
                        list.Add(value[0]);
                    }
                }
                if (CheckNextLine(line))
                {
                    retmode = DoxygenConfParseMode.DOXYGEN_PARSE_INPUT;
                }
            }
            return retmode;
        }

        private static DoxygenConfParseMode ParseSecondLine(DoxygenConfParseMode mode, string line, string key, ref List<string> list)
        {
            // 値の切り出し
            // 次行確認
            DoxygenConfParseMode retmode = mode;
            bool clear;
            int valueStart = GetValueStart(out clear, line);
            if (valueStart >= 0)
            {
                if (clear)
                {
                    list.Clear();
                }
                string[] value = GetValueName(line, valueStart);
                if (value.Length > 0)
                {
                    list.Add(value[0]);
                }
            }
            if (!CheckNextLine(line))
            {
                retmode = DoxygenConfParseMode.DOXYGEN_PARSE_OTHER;
            }
            return retmode;
        }

        //MEMO: コマンド記述後のコメントもサポートした方がいい？
        private static bool CheckFirstLine(string line)
        {
            string trimline = line.Trim();
            // 先頭が#ならばfalse
            if (trimline.StartsWith("#"))
            {
                return false;
            }
            // 空白ならばfalse
            // 次の行が続きであることを示す文字しかない場合はfalse
            if (trimline == string.Empty || trimline == "\\")
            {
                return false;
            }

            // 前の行からの続きであるならばfalseにしたいがそれは前の行で判断しているはず

            // 単語が存在するならばtrue (ただし=ではないこと)
            if (trimline != string.Empty && !trimline.StartsWith("="))
            {
                // TODO: 演算子が存在しないなら警告を出力する
                return true;
            }

            return false;
        }

        private static bool CheckNextLine(string line)
        {
            string trimline = line.Trim();
            if (trimline.EndsWith("\\"))
            {
                return true;
            }
            return false;
        }

        private static string GetKeyName(string line)
        {
            char[] separator1 = { ' ' };
            char[] separator2 = { '+', '=' };
            string[] words = line.Split(separator1, StringSplitOptions.RemoveEmptyEntries);
            if (words.Length == 0)
            {
                return string.Empty;
            }
            string[] words2 = words[0].Split(separator2, StringSplitOptions.RemoveEmptyEntries);
            if (words2.Length > 0)
            {
                return words2[0];
            }

            return string.Empty;
        }

        private static int GetValueStart(out bool clear, string line)
        {
            char[] separator1 = { ' ' };
            string[] words = line.Split(separator1, StringSplitOptions.RemoveEmptyEntries);
            clear = false;
            // "="があるならばその直後
            if (words.Contains("="))
            {
                clear = true;
                int index = line.IndexOf("=") + 1;
                return (index < line.Length) ? index : -1;
            }
            else if (words.Contains("+="))
            {
                int index = line.IndexOf("+=") + 2;
                return (index < line.Length) ? index : -1;
            }
            // そうでないならば行の先頭 (0にしておく
            return 0;
        }

        private static string[] GetValueName(string line, int start)
        {
            TextElementEnumerator charEnum = StringInfo.GetTextElementEnumerator(line.Substring(start).Trim());
            StringBuilder valuneName = new StringBuilder();
            List<string> valueNames = new List<string>();
            bool isQuote = false;
            while (true)
            {
                if (!charEnum.MoveNext())
                {
                    break;
                }
                if (isQuote)
                {
                    if (charEnum.GetTextElement() != "\"")
                    {
                        valuneName.Append(charEnum.Current);
                    }
                    else
                    {
                        valueNames.Add(valuneName.ToString());
                        valuneName.Clear();
                    }
                }
                else
                {
                    switch (charEnum.GetTextElement())
                    {
                        case "\\":
                            // "\\"ならば終端なので終わる
                            if (valuneName.Length > 0)
                            {
                                valueNames.Add(valuneName.ToString());
                                valuneName.Clear();
                            }
                            return valueNames.ToArray();
                        case "\"":
                            isQuote = true;
                            break;
                        case " ":
                            if (valuneName.Length > 0)
                            {
                                valueNames.Add(valuneName.ToString());
                                valuneName.Clear();
                            }
                            break;
                        default:
                            valuneName.Append(charEnum.Current);
                            break;
                    }
                }
            }
            if (valuneName.Length > 0)
            {
                valueNames.Add(valuneName.ToString());
            }

            return valueNames.ToArray();
        }
    }
}
