﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.DotNetLocalizer
{
    /// <summary>
    /// DllGeneratorクラスのExcecuteを実行するクラスです。
    /// </summary>
    public sealed class DllGeneratorInvoker : MarshalByRefObject
    {
        private readonly ResolveEventHandler currentDomainAssemblyResolveHandler = null;

        private LocalizationItem[] _localizeItems = null;

        public static void Invoke(
            string applicationBase,
            string privateBinPath,
            bool fillDllWithTemporary,
            LocalizationItem[] strRess)
        {
            AppDomain newDomain = null;
            try
            {
                newDomain = AppDomainUtility.CreateAppDomain(applicationBase, privateBinPath);
                var guiLocalizerApp = AppDomainUtility.CreateInstance<DllGeneratorInvoker>(newDomain);

                guiLocalizerApp.SetupSerializer(true);
                guiLocalizerApp.SetParamForSerializer(strRess);
                guiLocalizerApp.SetupSerializer(false);
                // ホスティング・クラスのエントリポイントを呼び出す
                guiLocalizerApp.Execute(applicationBase, fillDllWithTemporary);
            }
            finally
            {
                if (null != newDomain)
                {
                    AppDomain.Unload(newDomain);
                }
            }
        }

        public DllGeneratorInvoker()
        {
            currentDomainAssemblyResolveHandler = new ResolveEventHandler(CurrentDomain_AssemblyResolve);
        }

        /// <summary>
        /// シリアライズ用のセットアップを行います。
        ///
        /// ローカライズ対象のアセンブリを読み込むため、ローカライズ対象のアセンブリがある場所をアプリケーションベースとして、
        /// AppDomain.CreateInstanceFromAndUnwrap() を使用して AppDomain を作成しますが、このとき、
        /// アセンブリは Assembly.LoadFrom() にて読み込み元コンテキストに読み込まれます。
        ///
        /// AppDomain へ引数を渡すとき、MarshalByRefObject でないオブジェクトはバイナリシリアライズによって渡されますが、
        /// 読み込み元コンテキストに読み込まれているため、逆シリアル化のときに読み込みに失敗します。
        ///
        /// そのため、AssemblyResolve イベント時に読み込み済みのアセンブリを返すようにします。
        /// </summary>
        /// <param name="bInit"></param>
        private void SetupSerializer(bool bInit)
        {
            if (bInit)
            {
                AppDomain.CurrentDomain.AssemblyResolve += currentDomainAssemblyResolveHandler;
            }
            else
            {
                AppDomain.CurrentDomain.AssemblyResolve -= currentDomainAssemblyResolveHandler;
            }
        }

        /// <summary>
        /// GACに無いアセンブリにある型を使用するパラメータを設定します。
        /// </summary>
        /// <param name="localizeItems"></param>
        private void SetParamForSerializer(LocalizationItem[] localizeItems)
        {
            _localizeItems = localizeItems;
        }

        /// <summary>
        /// サテライトアセンブリ作成を実行します。
        /// </summary>
        private void Execute(string applicationBase, bool fillDllWithTemporary)
        {
            var dllGenerator = new DllGenerator(fillDllWithTemporary);
            dllGenerator.Execute(applicationBase, _localizeItems);
        }

        private Assembly CurrentDomain_AssemblyResolve(object sender, ResolveEventArgs args)
        {
            return typeof(LocalizationItem).Assembly;
        }
    }
}
