﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Reflection.Emit;
using System.Resources;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Windows.Markup.Localizer;
using Microsoft.Win32;

namespace Nintendo.DotNetLocalizer
{
    /// <summary>
    /// サテライトアセンブリ生成クラス。
    /// </summary>
    public sealed class DllGenerator
    {
        /// <summary>
        /// .resources ファイルで翻訳対象ではないリソース
        /// </summary>
        private static readonly Regex ExcludeResource =
            new Regex("^>>", RegexOptions.IgnoreCase | RegexOptions.Compiled);

        // 未翻訳箇所に仮文字列を挿入するかどうか。
        private static bool _fillDllWithTemporary = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DllGenerator(bool fillDllWithTemporary)
        {
            _fillDllWithTemporary = fillDllWithTemporary;
        }

        /// <summary>
        /// .resources ファイルで翻訳対象ではないリソースかどうかを判定します。
        /// </summary>
        /// <param name="key">リソースのキー。</param>
        /// <returns>翻訳対象でないリソースの場合は真。</returns>
        public static bool IsExcludeResourceKey(string key)
        {
            return ExcludeResource.IsMatch(key);
        }

        public static bool IsResourceEntryBamlStream(string name, object value)
        {
            string extension = Path.GetExtension(name);
            if (string.Compare(extension, ".baml", true) == 0)
            {
                Type type = value.GetType();

                if (typeof(Stream).IsAssignableFrom(type))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 実行。
        /// </summary>
        public void Execute(string asmDir, LocalizationItem[] allLocItems)
        {
            var locItemAsmGPs = allLocItems.GroupBy(res => res.AssemblyName, StringComparer.OrdinalIgnoreCase);
            foreach (var locItem in locItemAsmGPs)
            {
                GenerateEnResources(locItem.Key, locItem);
            }
        }

        private void GenerateEnResources(string asmName, IEnumerable<LocalizationItem> locItemsGpByAsm)
        {
            var assembly = Assembly.Load(asmName);
            var outputEnDir = Path.Combine(Path.GetDirectoryName(assembly.Location), "en");

            var outResDllName = asmName + ".resources.dll";

            var moduleLocalName = outResDllName;

            // AssemblyFileVersionAttribute のコピー。
#if false
            var customAttributeBuilders =
                from data in assembly.CustomAttributes
                where data.AttributeType == typeof(AssemblyFileVersionAttribute)
                let verStr = assembly.GetName().Version.ToString()
                select new CustomAttributeBuilder(
                        typeof(AssemblyVersionAttribute).GetConstructor(new Type[] { typeof(string) }),
                        new string[] { verStr });
#endif

            var targetAssemblyBuilder =
                AppDomain.CurrentDomain.DefineDynamicAssembly(
                    new AssemblyName(Path.GetFileNameWithoutExtension(outResDllName))
                    {
                        CultureInfo = new System.Globalization.CultureInfo("en"),
                        Version = assembly.GetName().Version
                    },
                    AssemblyBuilderAccess.Save, // AssemblyBuilderAccess.RunAndSave,
                    outputEnDir
#if false
                    ,
                    false,
                    customAttributeBuilders
#endif
                    );

            var moduleBuilder = targetAssemblyBuilder.DefineDynamicModule(moduleLocalName, outResDllName);

            // ResourceFile の "foo.resoureces:bar.baml" となっているのを、
            // Resources = "foo.resoureces", Baml = "bar.baml" と分ける。
            var items =
                from item in locItemsGpByAsm
                let files = item.ResourceFile.Split(new char[] { ':' }, 2, StringSplitOptions.RemoveEmptyEntries)
                select new { Resources = files[0], Baml = files.Length > 1 ? files[1] : string.Empty, Item = item };

            // .resources のファイル名でグループ化
            var locItemGpByResFiles =
                items.GroupBy(
                    tp => tp.Resources,
                    StringComparer.OrdinalIgnoreCase);

            var bAddResources = false;

            foreach (var locItem in locItemGpByResFiles)
            {
                // .resourcesから.en.resourcesを生成

                var resourceFileName = locItem.Key + ".resources";
                var resourceEnName = Path.ChangeExtension(resourceFileName, ".en.resources");
                Output.DebugWriteLine(" {0} - ", resourceFileName);

                // 最初のリソースを追加するまで遅延させる。
                var resWriter = new Lazy<IResourceWriter>(
                    () =>
                        moduleBuilder.DefineResource(
                            resourceEnName,
                            resourceEnName,
                            ResourceAttributes.Public));

                // baml でない翻訳対象の文字列の処理。
                var noBamlResEnum = locItem.Where(itemTp => itemTp.Baml == string.Empty);

                foreach (var itemTp in noBamlResEnum)
                {
                    resWriter.Value.AddResource(itemTp.Item.ResourceName, Utility.LFToCRLF(GetEnglishText(itemTp.Item.Text)));
                }

                using (var resReader = new ResourceReader(assembly.GetManifestResourceStream(resourceFileName)))
                {
                    // baml の翻訳対象の文字列の処理。
                    var enumLocItemGpByBaml =
                        locItem.Where(  // baml ファイルのみに絞り込む
                            itemTp => itemTp.Baml != string.Empty)
                        .GroupBy(       // bamlファイル名でグループ化
                            itemTp => itemTp.Baml,
                            itemTp => itemTp.Item,
                            StringComparer.OrdinalIgnoreCase)
                        .Join(          // bamlファイル名が一致するリソースをアセンブリから取得する
                            resReader.Cast<DictionaryEntry>(),
                            resItem => resItem.Key,
                            entry => entry.Key.ToString(),
                            (resItem, entry) => new { BamlGp = resItem, Stream = (Stream)entry.Value },
                            StringComparer.OrdinalIgnoreCase);
                    foreach (var bamlTp in enumLocItemGpByBaml)
                    {
                        // 日本語のbamlリソースを元に英語用bamlリソースを作成。
                        var bamlStream = TranslationBamlToBaml(bamlTp.BamlGp, bamlTp.Stream);
                        if (bamlStream != null)
                        {
                            resWriter.Value.AddResource(bamlTp.BamlGp.Key, bamlStream);
                        }
                    }
                }

                bAddResources = bAddResources || resWriter.IsValueCreated;

                // resWriter の Generate(), Close() は　AssemblyBuilder.Save() で
                // 呼ばれるため呼び出してはいけない。

                Output.DebugWriteLine(
                    " {0} -> {1}",
                    resourceFileName,
                    resourceEnName);
            }

            if (bAddResources)
            {
                Directory.CreateDirectory(outputEnDir);
                targetAssemblyBuilder.DefineVersionInfoResource();
                targetAssemblyBuilder.Save(outResDllName);
            }
        }

        /// <summary>
        /// .bamlを.resourcesに変換。
        /// </summary>
        private Stream TranslationBamlToBaml(IGrouping<string, LocalizationItem> locItems, Stream bamlStream)
        {
            // BamlLocalizer の中で、「pack:」スキーマを使った Uri コンストラクタが
            // 失敗しないようにするため PackUriHelper のスタティックコンストラクタが
            // 呼ばれるようにしておきます。
            var uriSchemePack = System.IO.Packaging.PackUriHelper.UriSchemePack;

            var bamlFileName = locItems.Key;
            Output.DebugWrite(" {0} - ", bamlFileName);

            //-----------------------------------------------------------------
            // .en.bamlを作成
            //-----------------------------------------------------------------
            BamlLocalizer bamlLocalizer = null;
            try
            {
                bamlLocalizer = new BamlLocalizer(bamlStream);
            }
            catch (Exception e)
            {
                Output.Error(e.Message);
                Utility.TestError(
                    false,
                    "関連するプロジェクトがビルドされているかどうか、確認してください。");
            }

            // 翻訳対象文字列を処理する
            var translations = new BamlLocalizationDictionary();

            var enumEnBamlResPair =
                from locItem in locItems
                join bamlResPair in bamlLocalizer.ExtractResources().Cast<DictionaryEntry>()
                    on locItem.ResourceName equals AssemblyUtility.GetKeyString((BamlLocalizableResourceKey)bamlResPair.Key)
                let bamlRes = (BamlLocalizableResource)bamlResPair.Value
                select new
                {
                    Key = (BamlLocalizableResourceKey)bamlResPair.Key,
                    Value =
                        new BamlLocalizableResource(
                            GetEnglishText(locItem.Text),
                            bamlRes.Comments,
                            bamlRes.Category,
                            bamlRes.Modifiable,
                            bamlRes.Readable)
                };

            foreach (var pair in enumEnBamlResPair)
            {
                translations.Add(pair.Key, pair.Value);
            }

            MemoryStream bamlEnStream = null;
            if (translations.Count > 0)
            {
                bamlEnStream = new MemoryStream();
                bamlLocalizer.UpdateBaml(bamlEnStream, translations);
            }
            Output.DebugWriteLine("->.en.baml");
            return bamlEnStream;
        }

        private string GetEnglishText(string text)
        {
            return
                string.IsNullOrEmpty(text) ?
                    (_fillDllWithTemporary ?
                        "[N/L]" :                   // 仮文字列を設定
                        string.Empty) :
                    text;
        }
    }
}
