﻿namespace DotNetAssemblyAttributeWriter
{
    using DotNetAssemblyAttributeWriter.Actions;
    using Nintendo.Foundation.Contracts;
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;

    /// <summary>
    /// エントリーポイントを持つプログラムクラスです。
    /// </summary>
    internal class Program
    {
        private const string DefaultActionFileName = "DefaultAction.yml";
        private const string AssemblyInfoFileName = "AssemblyInfo.cs";
        private const string SigloRootMarkFileName = "SigloRootMark";
        private const string SigloRootVariableName = "$(SigloRoot)";

        private static readonly string SigloRootDirectory = FindSigloRootDirectory();

        /// <summary>
        /// メイン関数（エントリーポイント）です。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        /// <returns>成功したら 0、失敗したら 1 を返します。</returns>
        public static int Main(string[] args)
        {
            try
            {
                var actionRoot = LoadActionFile(GetDefaultActionFilePath());

                if (actionRoot != null)
                {
                    actionRoot.Actions.ForEach(action => ExecuteAction(action));
                }
                else
                {
                    Console.WriteLine("action file not found.");
                }
            }
            catch
            {
                return 1;
            }

            return 0;
        }

        public static string GetDefaultActionFilePath()
        {
            return Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), DefaultActionFileName);
        }

        public static ActionRoot LoadActionFile(string filePath)
        {
            var deserializer = new YamlDotNet.Serialization.Deserializer();

            using (var reader = new StreamReader(filePath, Encoding.UTF8, true))
            {
                return deserializer.Deserialize<ActionRoot>(reader);
            }
        }

        public static void ExecuteAction(UpdateAssemblyInfoAction action)
        {
            Assertion.Argument.NotNull(action);
            Assertion.Argument.StringIsNotNullOrEmpty(action.TargetPath);
            Assertion.Argument.StringIsNotNullOrEmpty(action.CompanyName);
            Assertion.Argument.StringIsNotNullOrEmpty(action.CopyrightName);

            // "$(SigloRoot)" 変数の解決
            var targetFilePath = action.TargetPath.Replace(SigloRootVariableName, SigloRootDirectory);

            // ログ出力
            Console.WriteLine(
                string.Format(
                    "UpdateAssemblyInfo: \n  Directory={0}\n  CompanyName={1}\n  CopyrightName={2}",
                    targetFilePath,
                    action.CompanyName,
                    action.CopyrightName
                    )
                );

            // ファイル検索してアクションを実行
            IEnumerable<string> targetFilePaths = new string[0];

            if (Directory.Exists(targetFilePath))
            {
                targetFilePaths = Directory.EnumerateFiles(targetFilePath, AssemblyInfoFileName, SearchOption.AllDirectories);
            }
            else if (File.Exists(targetFilePath))
            {
                targetFilePaths = new string[] { targetFilePath };
            }

            targetFilePaths.ForEach(
                filePath => UpdateAssemblyInfo(filePath, action.CompanyName, action.CopyrightName)
                );

            // 空行を出力
            Console.WriteLine();
        }

        public static void UpdateAssemblyInfo(string filePath, string companyName, string copyrightName)
        {
            Console.WriteLine(filePath);
            AssemblyInfoWriter.UpdateAssemblyInfo(filePath, companyName, copyrightName);
        }

        public static string FindSigloRootDirectory()
        {
            var currentPath = Path.GetFullPath(Assembly.GetExecutingAssembly().Location);

            if (!Directory.Exists(currentPath))
            {
                currentPath = Path.GetDirectoryName(currentPath);
            }

            while (!string.IsNullOrEmpty(currentPath) && Path.IsPathRooted(currentPath))
            {
                if (File.Exists(currentPath + "\\" + SigloRootMarkFileName))
                {
                    break;
                }

                currentPath = Path.GetDirectoryName(currentPath);
            }

            return currentPath;
        }
    }
}
