﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//コード整形及び文字コードチェックツール
//
// 使い方
//  pre-commitファイル及びビルドして生成されるCommitContentChecker.exeを.git/hooksにコピーしてください。

using Nintendo.ToolFoundation.IO;
using CodingCheckerUtil;
using System;
using System.Reflection;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Collections;
using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("CommitContentCheckerTest")] // TODO: 厳密名の指定を検討する

namespace CommitContentChecker
{
    internal class Program
    {
        //----------------------------------------------
        //メイン関数
        internal static void Main(string[] args)
        {
            //コンソール出力をUTF8へ変更 TortoiseGitで問題が出るため保留
            //todo:出力の文字化け解消
            //Console.OutputEncoding = new UTF8Encoding();

            // コマンドライン引数を解析
            var parseResult = ParseCommandLine(args);
            if (parseResult.Params == null)
            {
                Environment.Exit(parseResult.ExitCode);
                return;
            }

            var @params = parseResult.Params;

            //git diffによりステージされたファイルの一覧を取得
            var gitExecuter = new GitExecuter(@params.GitPath);

            List<string> stagedFiles = new List<string>();
            try
            {
                stagedFiles.AddRange(gitExecuter.GetStagedFiles(@params.Against));
            }
            catch (System.Exception exception)
            {
                Console.Error.WriteLine("Error: Failed to read stage");
                Console.Error.WriteLine(exception);
                Environment.Exit(1);
            }

            TargetExtensionListParser targetExtensionListParser = null;
            try
            {
                 targetExtensionListParser = new TargetExtensionListParser(@params.TargetExtensionListFile);
            }
            catch (Exception)
            {
                Environment.Exit(1);
            }

            // Stage された変更内容が削除であるファイルは除外する
            stagedFiles = stagedFiles.Where(file => gitExecuter.GetIndexStatus(file) != 'D').ToList();

            // ファイルパスのチェック
            bool filePathCheckerResult = stagedFiles.All(file => FilePathChecker.CheckAll(file));

            // コードのチェック・自動整形
            var CommitContentChecker = new CommitContentChecker(@params.GitPath, targetExtensionListParser);
            bool CommitContentCheckerResult = stagedFiles.All(file => CommitContentChecker.CheckCode(file));

            if (!(filePathCheckerResult && CommitContentCheckerResult))
            {
                //一つでもファイルシステム上のパスと異なるパス、ファイル読み込み・書き込みエラー、UTF-8以外の文字コードがあった場合コミットしない
                Environment.Exit(1);
            }
            Environment.Exit(0);
        }

        private static ParseCommandLineResult ParseCommandLine(string[] args)
        {
            var s = new CommandLineParserSettings()
            {
                ApplicationDescription = "Validates what is going to be git-commit'ed.",
            };

            try
            {
                CommitContentCheckerParams @params;
                if (!new CommandLineParser(s).ParseArgs(args, out @params))
                {
                    return new ParseCommandLineResult(0);
                }
                return new ParseCommandLineResult(@params);
            }
            catch (ArgumentException)
            {
                return new ParseCommandLineResult(1);
            }
        }

        private class ParseCommandLineResult
        {
            public CommitContentCheckerParams Params { get; }
            public int ExitCode { get; }

            public ParseCommandLineResult(CommitContentCheckerParams @params)
            {
                this.Params = @params;
            }

            public ParseCommandLineResult(int exitCode)
            {
                this.Params = null;
                this.ExitCode = exitCode;
            }
        }
    }
}
