﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CodingCheckerUtil
{
    public static class GitUtil
    {
        private static readonly char[] Separators = new char[] { '\t', ' ' };

        /// <summary>
        /// tr からすべての行を読み取って返します。
        /// </summary>
        public static string[] ReadAllLines(TextReader tr)
        {
            var lines = new List<string>();

            string line;
            while ((line = tr.ReadLine()) != null)
            {
                lines.Add(line);
            }

            return lines.ToArray();
        }

        /// <summary>
        /// '"' から始まるエスケープされた文字列をハンドリングして返します。
        /// </summary>
        public static string Unquote(string s)
        {
            if (s.StartsWith("\""))
            {
                var list = new List<byte>();
                var i = 1;
                while (i < s.Length - 1)
                {
                    if (s[i] == '\\')
                    {
                        // \ で始まる場合には \ に続けて 3 ケタの 8 進数が続く
                        list.Add((byte)Convert.ToInt32(s.Substring(i + 1, 3), 8));
                        i += 4;
                    }
                    else
                    {
                        // \ で始まらない場合には ASCII 文字の範囲として読める
                        list.Add((byte)s[i]);
                        i += 1;
                    }
                }
                return Encoding.UTF8.GetString(list.ToArray());
            }
            else
            {
                return s;
            }
        }

        /// <summary>
        /// git ls-remote の出力をパースします。
        /// </summary>
        public static IReadOnlyCollection<RefInfo> ParseLsRemoteOutput(string output)
        {
            var refInfos = new List<RefInfo>();
            using (var sr = new StringReader(output))
            {
                var lines = GitUtil.ReadAllLines(sr);

                foreach (var line in lines)
                {
                    // フォーマットは次の通り:
                    // <sha1> \t <ref> LF
                    var cols = line.Split(Separators, options: StringSplitOptions.RemoveEmptyEntries);
                    if (cols.Length != 2)
                    {
                        continue;
                    }
                    refInfos.Add(new RefInfo(cols[1], cols[0]));
                }
            }
            return refInfos;
        }

        /// <summary>
        /// git status --porcelain の出力をパースして、index 上での status と work tree 上での status の値の組を返します。
        /// </summary>
        public static Tuple<char, char> ParseStatusOutput(string output)
        {
            using (var sr = new StringReader(output))
            {
                var line = sr.ReadToEnd();
                // フォーマットは次の通り:
                // <index status><work tree status> <filepath>
                string status = line.Substring(0, 2);

                return new Tuple<char, char>(status[0], status[1]);
            }
        }
    }
}
