﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CodingCheckerUtil
{
    public class GitExecuter
    {
        private readonly string _gitPath;

        public GitExecuter(string gitPath)
        {
            this._gitPath = gitPath;
        }

        public IEnumerable<string> GetStagedFiles(string against)
        {
            using (var sr = new StringReader(Execute($"diff --cached --name-only {against}")))
            {
                return GitUtil.ReadAllLines(sr).Select(x => GitUtil.Unquote(x));
            }
        }

        public byte[] GetStagedFileContent(string filename)
        {
            // SIGLO-71635: git show :"!.nact" は、次のエラーになる: fatal: ambiguous argument ':!.nact': unknown revision or path not in the working tree.
            // パスの前に ./ を付加することにより、パスとして解釈されるようにする。
            return ExecuteForBytes($"show :\"./{filename}\"");
        }

        public char GetIndexStatus(string filename)
        {
            try
            {
                return GitUtil.ParseStatusOutput(Execute($"status --porcelain \"{filename}\"")).Item1;
            }
            catch (SystemException e)
            {
                throw new GitExecuterException($"error: failed to get the status of file \"{filename}\" : {e.Message}");
            }
        }

        public char GetWorkTreeStatus(string filename)
        {
            try
            {
                return GitUtil.ParseStatusOutput(Execute($"status --porcelain \"{filename}\"")).Item2;
            }
            catch (SystemException e)
            {
                throw new GitExecuterException($"error: failed to get the status of file \"{filename}\" : {e.Message}");
            }
        }

        public IReadOnlyCollection<RefInfo> LsRemote(string remote)
        {
            return GitUtil.ParseLsRemoteOutput(Execute($"ls-remote {remote}"));
        }

        public void Add(string filename)
        {
            Execute($"add \"{filename}\"");
        }

        public void Reset(string filename)
        {
            Execute($"reset \"{filename}\"");
        }

        private string Execute(string args)
        {
            return Console.OutputEncoding.GetString(
                ExecuteForBytes(args));
        }

        private byte[] ExecuteForBytes(string args)
        {
            var psi = new ProcessStartInfo()
            {
                FileName = _gitPath,
                Arguments = args,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardInput = true,
                RedirectStandardOutput = true,
                RedirectStandardError = true,
            };

            try
            {
                using (var p = Process.Start(psi))
                {
                    byte[] ret;
                    string stderr;
                    using (var ms = new MemoryStream())
                    {
                        p.StandardInput.Close();
                        var outputTask = p.StandardOutput.BaseStream.CopyToAsync(ms);
                        var errorTask = p.StandardError.ReadToEndAsync();
                        p.WaitForExit();

                        outputTask.Wait();
                        errorTask.Wait();
                        ret = ms.ToArray();
                        stderr = errorTask.Result;
                    }

                    if (p.ExitCode != 0)
                    {
                        throw new GitExecuterException($"error: {_gitPath} {args} returned {p.ExitCode}. {stderr}");
                    }

                    return ret;
                }
            }
            catch (Win32Exception e)
            {
                throw new GitExecuterException($"error: failed to execute {_gitPath} {args}: {e.Message}");
            }
        }
    }

    [Serializable]
    public class GitExecuterException : Exception
    {
        public GitExecuterException() { }
        public GitExecuterException(string message) : base(message) { }
        public GitExecuterException(string message, Exception inner) : base(message, inner) { }
        protected GitExecuterException(
            System.Runtime.Serialization.SerializationInfo info,
            System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
    }
}
