﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace CodeChecker
{
    static internal class CodeChecker
    {
        static internal bool CheckCode(string file, TargetExtensionListParser targetExtensionListParser)
        {
            var checkOptions = targetExtensionListParser.GetFileCheckOptions(file);

            // ファイルがチェック対象ではない場合は何もしない
            if (checkOptions == CheckOptions.None)
            {
                return true;
            }

            // BOM チェック
            byte[] codeBytes;
            try
            {
                codeBytes = File.ReadAllBytes(file);
            }
            catch (Exception e) when (e is IOException || e is UnauthorizedAccessException)
            {
                Console.Error.WriteLine(file + ": Error: Failed to read file");
                Console.Error.WriteLine(e.Message);
                return false;
            }

            bool existsBOM = FormatterUtil.CheckBOM(codeBytes, checkOptions.HasFlag(CheckOptions.CheckBOM), $"{file}: ");
            if (checkOptions.HasFlag(CheckOptions.CheckBOM) && !existsBOM)
            {
                return false;
            }

            // 自動整形対象ではないファイルに対してはコーディングチェックを行わない
            if (!checkOptions.HasFlag(CheckOptions.AutoFormatContents))
            {
                return true;
            }

            // コーディングチェック
            var encoding = new UTF8Encoding(existsBOM);
            // BOM 付きの場合は BOM のバイト列は無視してデコード
            string codeContent = encoding.GetString(codeBytes, encoding.GetPreamble().Length, codeBytes.Length - encoding.GetPreamble().Length);
            if (ContentFormatter.Format(ref codeContent, true, $"{file}: ").IsFormatted)
            {
                return false;
            }
            return true;
        }
    }
}
