﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;
using System.IO;

namespace ApiLinkRewriter
{
    /**
     * <summary>Api リンクの情報の配置</summary>
     */
    class ApiLinkArranger
    {
        static readonly string regexText = "<span class=\"(?<apilink>ApiLink.*?)\">";

        /**
         * <summary> html ファイルに apilink タグがあればその Url リンクを付加する </summary>
         * <param name = "apiInfo">xml で読み出した Api 情報</param>
         * <param name = "fPath">html ファイルの絶対パス</param>
         * <param name = "saveAs">別ファイルに保存</param>
         * <param name = "logLevel">ログ表示レベルの指定</param>
         */
        public static void ArrangeHtmlHead(IEnumerable<ApiLink> apiInfo, string fPath, bool saveAs, OutputLog.LogLevel logLevel)
        {
            OutputLog.WriteDetail("::: Convert fileName ...", logLevel);

            // 出力する文字列
            var setLines = new StringBuilder();
            var regex = new Regex(regexText, RegexOptions.IgnoreCase);

            // UTF8 にエンコードしてファイル読み込み
            var htmltxt = File.ReadAllText(fPath, Encoding.GetEncoding("utf-8"));
            var matches = regex.Matches(htmltxt);
            // 正規表現の指定で取得したリンク群から重複しているものものを削除
            var linkTargets = (from Match match in matches select match.Groups["apilink"].Value).Distinct();

            foreach (var tagExpr in linkTargets)
            {
                // クラス名とパッケージ名に分解
                // タグ情報に含まれている Api リンクを収集していく
                var htmlLink = searchApiInfoForLink(apiInfo, separateApiTag(tagExpr));
                if (htmlLink == "")
                {
                    setLines.AppendFormat("// {0}", tagExpr);
                    setLines.Append(Environment.NewLine);
                }
                else
                {
                    setLines.AppendFormat("SetUrl( '{0}', '{1}' )", tagExpr, htmlLink);
                    setLines.Append(Environment.NewLine);
                }
            };
            // Api リンクが存在していれば JavaScript を付加
            if (linkTargets.Count() != 0)
            {
                var insertText = "";
                insertText += Texts.addLinkTop;
                insertText += setLines.ToString();
                insertText += Texts.addLinkBottom;
                htmltxt = htmltxt.Replace("</body>\r\n", insertText);
            }
            // 出力は上書きか別ファイルか
            var outputFile = saveAs ? fPath + ".output" : fPath;

            using (var sw = new StreamWriter(outputFile, false, Encoding.GetEncoding("utf-8")))
            {
                sw.Write(htmltxt);
            }
        }
        /**
         * <summary>Api タグをクラス名とパッケージ名に分割</summary>
         * <param name="tagString">Api タグ</param>
         * <returns>Api タグのクラス名とパッケージ名を持つクラス</returns>
         */
        private static ApiTag separateApiTag(string tagString)
        {
            var apiTag = new ApiTag();
            // クラス名とパッケージ名に分解。
            if (tagString.Contains(","))
            {
                var tagExprs = tagString.Split(',');
                apiTag.tagClass = tagExprs[0];
                apiTag.tagPackage = tagExprs[1];
            }
            else
            {
                apiTag.tagClass = tagString;
            }

            // クラス名を整形
            if (apiTag.tagClass.EndsWith("__"))
            {
                apiTag.tagClass = apiTag.tagClass.Substring(0, apiTag.tagClass.Length - 2);
            }

            return apiTag;
        }
        /**
         * <summary>タグ情報に含まれている Api リンクを収集していく</summary>
         * <param name="apiInfo">タグ情報</param>
         * <returns>Api タグのクラス名とパッケージ名を持つクラス</returns>
         * <returns>Api のリンクアドレス、一致するものがなければ空文字</returns>
         */
        private static string searchApiInfoForLink(IEnumerable<ApiLink> apiInfo, ApiTag apiTag)
        {
            // Api リンクがタグ情報に含まれていた場合は、最初に一致したリンクを返す
            var rules = new List<ApiLink>();
            foreach (var info in apiInfo)
            {
                if (info.name != apiTag.tagClass) continue;
                if (info.package != apiTag.tagPackage && apiTag.tagPackage != "") continue;
                rules.Add(info);
            }
            if (rules.Count == 0) return "";
            return rules.First<ApiLink>().html.ToString();
        }
    }
}
