<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Runs multiple tests at the same time as NxAgingHelperCommand.  These are stress tests that should run for a long time.

    .DESCRIPTION
        - Runs NxAgingHelperCommand and then run a list of tests at the same time
        
#>

param
(
    [string]
    # SDK root directory
    $sigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS | ? { $_ } | % { $_.Trim() }),
    
    [int]
    # Length of time to run in seconds
    $TimeToRunSeconds =
    ($env:TEST_NXAGING_TIME_SECONDS | ? { $_ } | % { $_.Trim() }),
    
    [string]
    # eg. NX/DisplayTest,NX/AudioTest
    # used for tests which don't include command --enable-power-button-event
    $TestListYml =
    ($env:TESTRUNNER_CONFIG_NAME),

    [string]
    # eg. Tests\Outputs\NX-NXFP2-a64\Tests\testEdid_unit\Develop\testEdid_unit.nsp,Tests\Outputs\NX-NXFP2-a64\Tests\testAudio_AudioOut\Develop\testAudio_AudioOut.nsp
    # used for tests which include command --enable-power-button-event
    $TestListApp =
    ($env:TESTRUNNER_EXE_LIST),
    
    [string]
    # eg. NxAgingConfig_Default.yml
    $NxAgingConfigFile =
    ($env:TEST_NXAGING_CONFIG_FILE),

    [string]
    # eg.  --enable-sdev-usb-event --enable-power-button-event
    $NxAgingCommands =
    ($env:TEST_NXAGING_COMMANDS)
)


# get the root path for the repo
$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# go to the root directory
Set-Location -Path $sigloRoot

# constants
$NX_AGING_WORKING_DIR = "$sigloRoot\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release"
$NX_AGING_EXE = "$sigloRoot\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release\NxAgingHelperCommand.exe"

# sleep/wake command needs special handling
$NX_AGING_COMMAND_SLEEP = "--enable-power-button-event"


# Run NxAgingHelperCommand.exe, and then the tests in parallel for a specified duration
$JobList = @{
    
    # NxAgingHelperCommand
    "NxAgingHelperCommand" = [scriptblock]::Create("$NX_AGING_EXE --config $sigloRoot\Integrate\Sources\Tools\NxAgingTools\Configuration\$NxAgingConfigFile --event-log-path $NX_AGING_WORKING_DIR\NxAgingHelper_Event.log --enable-uart-log --uart-log-path $NX_AGING_WORKING_DIR\NxAgingHelper_Uart.log --sdev $TargetAddress --time $TimeToRunSeconds $NxAgingCommands");
    
    # Test list
    "TestList" = {
        param($LocalSigloRoot,$LocalTimeToRunSeconds,$LocalNxAgingCommands,$LocalTestListApp,$LocalTestListYml,$LocalNX_AGING_COMMAND_SLEEP)

        &   {
            # this script block cannot access variables outside of it
            Set-Location -Path $LocalSigloRoot
            $timeToRunTestsSeconds = $LocalTimeToRunSeconds
            $stopWatch = [Diagnostics.Stopwatch]::StartNew()
            $elapsedTime = 0
            $elapsedTimeSeconds = 0
            while ($elapsedTimeSeconds -lt $timeToRunTestsSeconds)
            {
                if ($LocalNxAgingCommands -match $LocalNX_AGING_COMMAND_SLEEP)
                {
                    $testExeList = $LocalTestListApp -csplit ","
                    $testExeList | %{
                        $timeoutForRun = [math]::Round($timeToRunTestsSeconds - $elapsedTimeSeconds)
                        Write-Host "-- testlist: $_"
                        & Tools\CommandLineTools\RunOnTarget.exe --failure-timeout $timeoutForRun $_
                    }
                }
                else
                {
                    $testConfigList = $LocalTestListYml -csplit ","
                    $testConfigList | %{
                        Integrate/Scripts/Invoke-Script.ps1 Invoke-TestRunner $_
                    }
                }
                $elapsedTime = $stopWatch.Elapsed
                $elapsedTimeSeconds = $elapsedTime.TotalSeconds
            }
        }
    }
}


# Report the status of running jobs this many seconds
$POLL_TIME = 600

try {

    # if the NxAgingHelper command is for sleep/wake or --enable-power-button-event, enable logging on the device
    if ($NxAgingCommands -match $NX_AGING_COMMAND_SLEEP)
    {
        Write-Host "--- Starting sleep/wake"
        & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug enable-sd-card-logging
		& Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
        Start-Sleep -s 30
    }
    
    # Start all of the jobs
    $Jobs = $(
        ForEach ($item in $JobList.GetEnumerator()) {
            Write-Host "Starting $($item.Name) as $($item.Value)"
            Start-Job -Name $item.Name -ScriptBlock $item.Value -ArgumentList $sigloRoot,$TimeToRunSeconds,$NxAgingCommands,$TestListApp,$TestListYml,$NX_AGING_COMMAND_SLEEP
        }
    )

    # Wait for all the jobs to finish
    # We could just use "Wait-Job $Jobs", except that with no output for hours, TC might think the job is hung.
    While($true) {
        Wait-Job $Jobs -Timeout $POLL_TIME | Out-Null
        $Running = $($Jobs | Where-Object {$_.State -eq "Running"})
        if ($Running) {
            Write-Host "Waiting for jobs: $($Running | Select -expand Name)"
        } else {
            Write-Host "Jobs Completed"
            Break
        }
    }
}

# This is in a finally block, so that any remaining jobs will be stopped if this script is interrupted.
finally {
    Set-Location -Path $sigloRoot
    
    # The normal output should be redirected to a log file, but startup errors may be displayed here.
    # Use the teamcity macros to differentiate output from each command.
    $Jobs | % {
        $JobName = $_.Name
        Write-Host "##teamcity[testStarted name='$JobName' captureStandardOutput='true']`n"
        Stop-Job $_
        $Job = Receive-Job $_ -Keep -Verbose
        $Job = $Job -join "`n"
        Write-Host $Job
        if ($Job.Error) {
            Write-Host "##teamcity[testFailed name='$JobName']"
        }

        if ($Job.PSEndTime) {
            $duration = ($Job.PSEndTime - $Job.PSBeginTime).TotalMilliseconds
        } else {
            $duration = 0
        }
        Write-Host "##teamcity[testFinished name='$JobName' duration='$duration']"
    }
}

