﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Stop an unrelated build

    .DESCRIPTION
        Cancel a build when the build doesn't relate to specified modules.
#>

param
(
    [string]
    # The path to an authentication file
    $AuthenticationFile,

    [string]
    # The CI server url
    $Url,

    [string]
    # Specify the build id
    $BuildId,

    [string[]]
    # Specify the modules
    $Module,

    [string[]]
    # Specify the test categories
    $Category =
    ($env:NNTEST_CATEGORY | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() })
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\TeamCity"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

function Get-QuotedText([string]$text)
{
    try
    {
        $builder = New-Object "System.Text.StringBuilder"

        $reader = New-Object "System.IO.StringReader" $text
        while ($true)
        {
            $line = $reader.ReadLine()

            if ($null -eq $line) { break }

            [void]$builder.AppendLine("> ${line}")
        }

        return $builder.ToString()
    }
    finally
    {
        $reader.Dispose()
    }
}

function Get-WebExceptionMessage([System.Net.WebException]$ex)
{
    if ([System.Net.WebExceptionStatus]::ProtocolError -eq $ex.Status)
    {
        $responce   = [System.Net.HttpWebResponse]$ex.Response
        $statusCode = [int]$responce.StatusCode
        $statusName = $responce.StatusCode
        return "Received error responce (${statusCode} ${statusName})"
    }
    else
    {
        return "Failed to access the server ($($ex.Status))"
    }
}

if ([string]::IsNullOrEmpty($AuthenticationFile))
{
    throw "'AuthenticationFile' is required"
}

if ([string]::IsNullOrEmpty($Url))
{
    throw "'Url' is required"
}

if ([string]::IsNullOrEmpty($BuildId))
{
    throw "'BuildId' is required"
}

if ($null -eq $Module)
{
    throw "'Module' is required"
}

$Category | ? { "Basic" -ceq $_ } | % {
    Write-Host "This build belongs to basic test job"
    exit 0
}

$authFile = [xml](Get-Content $AuthenticationFile -ErrorAction Stop)
$userName = $authFile.Configuration.User
$password = $authFile.Configuration.Password

$changeList = & {
    try
    {
        $locator = @{ build = @{ id = $BuildId } }
        return (Get-TeamCityChangeList $Url $userName $password $locator)
    }
    catch [System.Net.WebException]
    {
        Write-ErrorMessage (Get-WebExceptionMessage $_.Exception)
        exit 0 # quiet exit
    }
}
if (0 -eq $changeList.changes.count)
{
    Write-Host "No changes exist"
    exit 0
}

$haystack = & {
    $moduleList = @{}

    $changeList.changes.change | % {
        $changeDetail = & {
            try
            {
                return (Get-TeamCityChangeDetail $Url $userName $password $_.id)
            }
            catch [System.Net.WebException]
            {
                Write-ErrorMessage (Get-WebExceptionMessage $_.Exception)
                exit 0 # quiet exit
            }
        }

        $comment = $changeDetail.change.comment
        switch -regex -casesensitive ($comment)
        {
            "^Automatic merge"               {}
            "^Merge branch "                 {}
            "^Merge pull request "           {}
            "^Merge remote-tracking branch " {}
            "^(?:product|other):\[.+?-\d+\] ?\((.+?)\)"
            {
                $matches[1].Split(",") | % { $moduleList[$_.Trim()] = $null }
            }
            default
            {
                $message  = "Unexpected comment detected:"
                $message += [System.Environment]::NewLine
                $message += Get-QuotedText $comment
                Write-ErrorMessage $message
                exit 0 # quiet exit
            }
        }
    }

    return $moduleList.Keys
}

if (0 -eq $haystack.Count)
{
    Write-Host "Module name not found in commit log"
    exit 0
}

Write-Host "needle(s): $($Module   -join ',')"
Write-Host "haystack : $($haystack -join ',')"
$Module | % {
    if ($haystack -contains $_)
    {
        Write-Host "Module '$_' was found in commit log"
        exit 0
    }
}
Write-Host "Trying to cancel this build..."
Set-EnvironmentVariableForTeamCity "NNTEST_IS_CANCELLED" "t"

exit 0
