<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Initialize sdev by system updater.

    .DESCRIPTION
        Initialize sdev by system updater.
#>
[CmdletBinding()]
Param
(
    # The host IP address pattern as RegularExpression
    [Parameter(Mandatory=$true)]
    [string]
    $AddressPattern = '.*',
    
    [string]
    $BuildType = 'Develop',
    
    [string]
    $Platform = 'NX-NXFP2-a32',
    
    [string]
    $Target,
    
    [string]
    $Options = '',
    
    [string]
    $InitialImage,

    [string]
    $KeyType = 'Auto', # K1, K2, Auto
    
    [string]
    $ConnectionType = 'Hb', # Hb, Usb
    
    [string]
    $BootType = 'Nand', # Nand, List
    
    [string]
    $SignedType = 'Unsigned', # Unsigned, Signed
    
    [string]
    $StorageSize = '64G', # 32G, 64G 
    
    [string]
    $DeveloperType = 'Internal', # Internal, Public    

    [Int]
    $DetectTargetRetryCount = 5,

    [string]
    $Suffix = ''
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/HostBridge"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/Runner"
Import-Module "${scriptDirectoryPath}/../../Modules/InitializeTarget"
Import-Module "${scriptDirectoryPath}/../../Modules/UsbTarget"

$SystemUpdater = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/$Platform/TargetTools/SystemUpdaterHostFs/$BuildType/SystemUpdaterHostFs.nsp"
$RunOnTarget =   "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RunOnTargetPrivate.exe"
$ControlTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$UpdateHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/Update-HostBridge.ps1"
$HelloWorld  = "$(Get-NintendoSdkRootPath)/Programs/Iris/Outputs/$Platform/TargetTools/HelloWorld/$BuildType/HelloWorld.nsp"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# Get the address of the HostBridge
$TargetAddress = Get-SigleTargetAddress -AddressPattern $AddressPattern

if([String]::IsNullOrEmpty($InitialImage))
{
    if($KeyType -eq 'Auto')
    {
        # Update HostBridge

        Invoke-CriticalCommand "& `"$UpdateHostBridge`" -AddressPattern `"$AddressPattern`""
        $KeyType = Get-TargetFusedKeyType $TargetAddress
    }

    $InitialImageName = "NX-$KeyType-$ConnectionType-$SignedType-$BootType-$StorageSize-$DeveloperType"
    $InitialImagePath =  "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/InitialImages/$InitialImageName/$BuildType/$InitialImageName.initimg"
}
else
{
    $InitialImagePath = $InitialImage
}

# Test existing files
Test-EnsureExistPath $SystemUpdater
Test-EnsureExistPath $RunOnTarget
Test-EnsureExistPath $ControlTarget
Test-EnsureExistPath $InitialImagePath
Test-EnsureExistPath $HelloWorld

if([String]::IsNullOrEmpty($Target))
{
    $TargetOption = "--target $TargetAddress --reset --wait-after-reset 10 "
}
elseif($Target -eq 'usb')
{
    # Detect target
    Write-Host "Find USB target device..."
    Get-SigleUsbTargetName -RetryCount $DetectTargetRetryCount  | Out-Null
    $TargetOption = "--target $Target "
}
else
{
    $TargetOption = "--target $Target --reset --wait-after-reset 10 "
}

# Write
Invoke-CriticalCommand "& `"$RunOnTarget`" run $TargetOption --verbose --monitor-serial --hostbridge $TargetAddress --failure-timeout 600 --pattern-success-exit `"Succeeded initializing the system.`" `"$SystemUpdater`" '--' --input `"$InitialImagePath`" $Options"
# HelloWorld (test booting)
Invoke-CriticalCommand "& `"$RunOnTarget`" run $TargetOption --reset --wait-after-reset 20 --verbose --monitor-serial --hostbridge $TargetAddress --failure-timeout 30 --pattern-success-exit `"Hello, world.`" `"$HelloWorld`" -- 5"
