﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Kernel Update

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [parameter(Mandatory=$true)]
    [string]
    # Target IP address
    $target,

    [string[]]
    # If failure key word detected, the script will fail
    $FailureKeyWords,

    [switch]
    # If "Break" log detect, the script will fail
    $FailureByBreak,

    [switch]
    # If reboot log detect, the script will fail
    $FailureByReboot
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Error"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"

$ResultDirectoryPath = "$(Get-NintendoSdkRootPath)\Integrate\Outputs\NxAgingTools"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# "grep $keyword $filename | wc -l" に相当する処理
function GrepAndCount()
{
    param
    (
        [parameter(Mandatory=$true)]
        [string]
        $FileName,

        [parameter(Mandatory=$true)]
        [string]
        $KeyWord
    )

    $a = Select-String -CaseSensitive $KeyWord $FileName
    if ($a -eq $null)
    {
        return 0
    }
    elseif ($a -is [Microsoft.PowerShell.Commands.MatchInfo])
    {
        return 1
    }
    elseif ($a -is [System.Object[]])
    {
        return $a.Length
    }
    else
    {
        throw "Select-String returns an unexpected type."
    }
}

$keywords = @();
if ($FailureKeyWords -is [string[]])
{
    foreach($_ in $FailureKeyWords)
    {
        $keywords += $_
    }
}
if ($FailureByBreak)
{
    $keywords += "Break"
}
if ($FailureByReboot)
{
    $keywords += "Nintendo NX bootloader"
}

$eventLogPath = "$ResultDirectoryPath\NxAgingTools_${target}_Event.log"
$uartLogPath = "$ResultDirectoryPath\NxAgingTools_${target}_Uart.log"

$ohayoCount = GrepAndCount -FileName $uartLogPath -KeyWord "OHAYO"
$oyasumiCount = GrepAndCount -FileName $uartLogPath -KeyWord "OYASUMI"
$rebootCount = GrepAndCount -FileName $uartLogPath -KeyWord "Nintendo NX bootloader"

Write-Host "Target: ${target}"
Write-Host "  - OHAYO:   ${ohayoCount}"
Write-Host "  - OYASUMI: ${oyasumiCount}"
Write-Host "  - Reboot:  ${rebootCount}"

$failure = $false
foreach($keyword in $keywords)
{
    $count = GrepAndCount -FileName $uartLogPath -KeyWord $keyword
    Write-Host "  - [failure keyword] `"${keyword}`": ${count}"
    if ($count -gt 0)
    {
        $failure = $true
    }
}

if ($failure)
{
    Write-Host "Failure log detected."
    exit 1
}
exit 0
