﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Error Module

    .DESCRIPTION
        This file defines functions for error handling
#>


function Write-ErrorMessageWithFormat([string]$path, [string]$message)
{
    $scriptName = [System.IO.Path]::GetFileNameWithoutExtension($path)
    $Host.UI.WriteErrorLine("[${scriptName}] Error: ${message}")
}

<#
    .SYNOPSIS
        Prints error message

    .PARAMETER Message
        Error message
#>
function Write-ErrorMessage
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$Message
    )

    Write-ErrorMessageWithFormat $MyInvocation.ScriptName $Message
}
Export-ModuleMember -Function Write-ErrorMessage

<#
    .SYNOPSIS
        Prints error record

    .PARAMETER Record
        Error record
#>
function Write-ErrorRecord
{
    param
    (
        [Parameter(Mandatory=$true)]
        [Management.Automation.ErrorRecord]$Record
    )

    $builder = New-Object System.Text.StringBuilder
    [void]$builder.Append($Record.Exception.Message)
    [void]$builder.Append(" (")
    if ( $Record.InvocationInfo.ScriptName )
    {
        $name = [System.IO.Path]::GetFileName($Record.InvocationInfo.ScriptName)
        [void]$builder.Append($name)
        [void]$builder.Append(", ")
    }
    [void]$builder.Append("Line ")
    [void]$builder.Append($Record.InvocationInfo.ScriptLineNumber)
    [void]$builder.Append(", Offset ")
    [void]$builder.Append($Record.InvocationInfo.OffsetInLine)
    [void]$builder.Append(")")
    Write-ErrorMessageWithFormat $MyInvocation.ScriptName $builder.ToString()
}
Export-ModuleMember -Function Write-ErrorRecord
