﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Sleep Test on Target

    .DESCRIPTION
        Run Sleep Test.
#>

param
(
    [string]
    # The path to test_detail.xml
    $TestDetailXmlPath,

    [int]
    # Number of repetitions.(-1 means repeating forever.)
    $RepeatCount = 3,

    [int]
    # Time of execution interval at repeat.(millisecond)
    $TimeOfRepeatInterval = 1000,

    [int]
    # Number of seconds before it times out.(second)
    $Timeout = 30,

    [int]
    # Time to first sleep.(second)
    $TimeOfAgingStartWait = 5,

    [int]
    $TimeOfSleepBeforeTest = 10,

    [string]
    # Test path
    $Path,

    [string]
    # Target Name
    $TargetName,

    [string]
    # Target Address
    $TargetAddress,

    [string]
    # Target Address Pattern
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN,

    [string]
    # The platform name
    $Platform = "NX64",

    [string]
    # The build type
    $BuildType = "Develop"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"
Import-Module "${scriptDirectoryPath}\Modules\SystemTestUtility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($AddressPattern))
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not
    # difined
    $AddressPattern = "^169\.*"
}

$VsPlatform = switch ($Platform)
{
    "NX32"      { "NX-NXFP2-a32" }
    "NXFP2-a32" { "NX-NXFP2-a32" }
    "NX64"      { "NX-NXFP2-a64" }
    "NXFP2-a64" { "NX-NXFP2-a64" }
    default     { $_ }
}

if ([string]::IsNullOrEmpty($Path))
{
    $Path = Get-NintendoSdkRootPath
    $Path = [IO.Path]::Combine($Path, "Integrate\Outputs")
    $Path = [IO.Path]::Combine($Path, "Scripts\Demo1Internal")
    $Path = [IO.Path]::Combine($Path, "Demo1Internal.$VsPlatform.$BuildType.nsp")
}

if ([string]::IsNullOrEmpty($TargetName))
{
    $TargetName = Get-SigleTargetAddress -AddressPattern $AddressPattern
    $TargetAddress = $TargetName
}

if (-not $TestDetailXmlPath)
{
    Write-ErrorMessage "TestDetailXmlPath option not specified"
    exit 1
}

$PressPowerButtonCommand="press-power-button-for-sc7"

# TORIAEZU :
#  dll が読めないので、暫定対応として、Outputs 以下の ControlTarget を実行
$ControlTarget = Get-NintendoSdkRootPath
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Programs\Chris\Outputs")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "x86")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Tools\RunnerTools")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Release")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate.exe")

$RunOnTarget = Get-NintendoSdkRootPath
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "Tools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "CommandLineTools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "RunOnTarget.exe")

$ReadHbLog = Get-NintendoSdkRootPath
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "Tools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "CommandLineTools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "ReadHostBridgeLog.exe")

$HidShell = Get-NintendoSdkRootPath
$HidShell = [IO.Path]::Combine($HidShell, "Programs\Iris\Outputs")
$HidShell = [IO.Path]::Combine($HidShell, "x64\Tools\Hid\HidShell\Release")
$HidShell = [IO.Path]::Combine($HidShell, "HidShell.exe")

$logDir = $scriptDirectoryPath
$logDir = $logDir.Replace(
    (Get-NintendoSdkIntegrateRootPath),
    (Get-NintendoSdkIntegrateOutputRootPath))
$logDir = [IO.Path]::Combine(
    $logDir, [IO.Path]::GetFileNameWithoutExtension($scriptPath))
$logDir = [IO.Path]::Combine($logDir, (Get-Date -Format "yyyyMMddHHmmss"))
[void](New-Item $logDir -ItemType Directory -ErrorAction "Stop")

$logFile = [IO.Path]::GetFileNameWithoutExtension($Path)
$logFile = [IO.Path]::Combine($logDir, ($logFile + ".log"))

Write-Host "Start ${Path}"
Write-Host "logFile: ${logFile}"

# FromHost 起動など、spsm が psc のモジュール登録を待たない場合は、
# すぐにテストを走らせるとモジュールが死ぬため待つ
Start-Sleep -Seconds $TimeOfSleepBeforeTest

# 全てのテストでのレジュームにかかる時間格納用の配列
$script:timeArrayAll = @()

# テスト対象プログラムの実行
function Invoke-TestProgram()
{
    # テストを起動 (実行直後に RunOnTarget は終了)
    $command  = ("&", "`"$RunOnTarget`"", "`"$Path`"")
    $command += ("--target", $TargetName)
    $command += ("--verbose", "--no-wait", "--failure-timeout", "$Timeout")
    $command += ("--suppress-auto-kill")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    $TestResult = $LastExitCode
    if (0 -ne $TestResult)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($Path))
    }

    return $TestResult
}

function Invoke-DoSleep()
{
    # スリープしたことを判定する為のログ
    $SleepLog = "OYASUMI"

    $list = ($ControlTarget, $TargetAddress, $PressPowerButtonCommand)
    $job = Start-Job -ArgumentList $list -ScriptBlock {
        param($path, $name, $cmd)
        $subCommand  = ("&", "`"$path`"", $cmd)
        $subCommand += ("--ip-addr", $name)
        $subCommand = $subCommand -join " "
        Invoke-Expression $subCommand
        $LastExitCode
    }

    # HostBridge Log を監視してスリープの完了を待つ
    #  下記のいずれかが発生した場合にエラー
    #  - タイムアウト時間(デフォルト30秒)内にスリープ遷移完了のログが出力されない
    #  - アサート関連ログが出力されている
    $command  = ("&", "`"$ReadHbLog`"")
    $command += ("--address", $TargetAddress)
    $command += ("--verbose", "--failure-timeout", "$Timeout")
    $command += ("--pattern-success-exit", "`"$SleepLog`"")
    $command += ("--pattern-failure-exit", "`"Assertion Failure:`"")
    $command += ("--pattern-failure-exit", "`"Break\(\) called`"")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    $TestResult = $LastExitCode

    # バックグラウンドジョブの完了待ち
    Wait-Job -job $job > $null

    # バックグラウンドジョブ結果取得
    $BgJobResult = Receive-Job -job $job
    if (0 -ne $BgJobResult)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
    }

    # 明示的にバックグラウンドジョブを停止／削除する
    Stop-Job $job > $null
    Remove-Job $job > $null

    return $TestResult
}

# スリープ/レジュームの繰り返しテストの実行
function Invoke-SleepResumeTest($repeatCount,$repeatTime)
{
    # レジュームしたことを判定する為のログ
    $ResumeLog = "nn::oe::MessageResume"

    # 単一のテストケースでのスリープ/レジュームにかかる時間格納用の配列
    $timeArray = @()

    # テストプログラム実行 (してすぐ RunOnTarget は終了)
    $TestResult = Invoke-TestProgram
    if (0 -ne $TestResult)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($Path))
    }

    # エージング試験を開始する前に指定時間(デフォルト5秒)待つ
    Start-Sleep -Seconds $TimeOfAgingStartWait

    $count = $repeatCount
    while($count -ne 0)
    {
        # スリープさせる
        $TestResult = Invoke-DoSleep
        if (0 -ne $TestResult)
        {
            # スリープが失敗したらテスト終了
            break
        }

        # 電源ボタン押下をエミュレーションする
        $command  = ("&", "`"$ControlTarget`"")
        $command += ($PressPowerButtonCommand)
        $command += ("--ip-addr", $TargetAddress)
        $command = $command -join " "
        Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
        $TestResult = $LastExitCode
        if (0 -ne $TestResult)
        {
            Invoke-Expression "& `"$ControlTarget`" reset --hard --target $TargetName"

            throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
        }

        # TargetManager に明示的に接続する
        Invoke-Expression "& `"$ControlTarget`" connect -t $TargetName"

        # InputDirector との接続を待つ
        Start-Sleep -Seconds $TimeOfSleepBeforeTest

        # バックグラウンドジョブで HostBridge Log を監視してレジュームの完了を待つ
        #  下記のいずれかが発生した場合にエラー
        #  - タイムアウト時間(デフォルト30秒)内にレジューム遷移完了のログが出力されない
        #  - アサート関連ログが出力されている
        $list = ($ReadHbLog, $TargetAddress, $Timeout, $ResumeLog)
        $job = Start-Job -ArgumentList $list -ScriptBlock {
            param($path, $name, $time, $log)
            $subCommand  = ("&", "`"$path`"")
            $subCommand += ("--address", $name)
            $subCommand += ("--verbose")
            $subCommand += ("--failure-timeout", $time)
            $subCommand += ("--pattern-success-exit", "`"$log`"")
            $subCommand += ("--pattern-failure-exit", "`"Assertion Failure:`"")
            $subCommand += ("--pattern-failure-exit", "`"Break\(\) called`"")
            $subCommand = $subCommand -join " "
            Invoke-Expression $subCommand
            $LastExitCode
        }

        # ホームボタン押下をエミュレーションする
        $command  = ("&", "`"$HidShell`"")
        $command += ("send-event")
        $command += ("-t", $TargetName)
        $command += ("HomeButton 0 ButtonDown")
        $command = $command -join " "
        Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
        $TestResult = $LastExitCode
        if (0 -ne $TestResult)
        {
            # ホームボタン押下が失敗したらテスト終了
            break
        }

        # ホームボタン解放をエミュレーションする
        $command  = ("&", "`"$HidShell`"")
        $command += ("send-event")
        $command += ("-t", $TargetName)
        $command += ("HomeButton 0 ButtonUp")
        $command = $command -join " "
        Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
        $TestResult = $LastExitCode
        if (0 -ne $TestResult)
        {
            # ホームボタン解放が失敗したらテスト終了
            break
        }

        # レジュームにかかる時間の計測開始
        $localWatch = New-Object "System.Diagnostics.StopWatch"
        $localWatch.Start()

        # バックグラウンドジョブの完了待ち
        Wait-Job -job $job > $null

        # バックグラウンドジョブ結果取得
        $BgJobResult = Receive-Job -job $job
        if ($true -ne ([string]$BgJobResult).Contains("Exit program. Status = Success"))
        {
            Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

            throw ("Failed to run " + [IO.Path]::GetFileName($ReadHbLog))

            # テストが失敗したら終了
            $TestResult = 1
            break
        }

        # 時間計測終了
        $localWatch.Stop()

        # 明示的にバックグラウンドジョブを停止／削除する
        Stop-Job $job > $null
        Remove-Job $job > $null

        # レジューム検出時間を配列にpush
        $timeArray += $localWatch.Elapsed.Milliseconds

        # 次のスリープ/レジューム実施時間まで待つ
        Start-Sleep -m $repeatTime

        # 繰り返し回数をデクリメント
        if($count -gt 0)
        {
            $count -= 1
        }
    }

    # ターゲットをリセットする
    $command  = ("&", "`"$ControlTarget`"", "reset")
    $command += ("--target", $TargetName)
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    if (0 -ne $LastExitCode)
    {
        throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
    }
    Start-Sleep -Seconds $TimeOfSleepBeforeTest

    $script:timeArrayAll += $timeArray
    # 平均時間をテスト実行時間とする
    $max,$min,$avg,$unbiasedVariance = Get-Statistic $timeArray
    if (0 -ne $TestResult)
    {
        return $false ,$avg
    }
    return $true ,$avg
}

# テストケース
$tests = New-Object "System.Collections.Specialized.OrderedDictionary"

# スリープ切り替えでDemo1が正常に動作するか
$tests["SleepTest"] =
{
    return Invoke-SleepResumeTest $RepeatCount $TimeOfRepeatInterval
}

# テストスイート名
$classname = "SleepResumeTestSuite"

# テスト実行
$xml = Start-TestSuite $classname $tests

# テスト結果出力
$xml.Save($TestDetailXmlPath)

# fail発生回数取得
$failures = $xml.testsuites.failures

# レジューム回数が２回以上の場合、レジュームにかかる時間の最大値・最小値・平均値・不偏分散をログ出力
if ($script:timeArrayAll.Length -ge 2)
{
    $max,$min,$avg,$unbiasedVariance = Get-Statistic $script:timeArrayAll

    # 結果の出力
    Write-Host "Resume measurement time,"
    Write-Host "Resume total num     = " $script:timeArrayAll.Length
    Write-Host "max                  = " $max ms
    Write-Host "min                  = " $min ms
    Write-Host "average              = " $avg ms
    Write-Host "unbiased variance    = " $unbiasedVariance
}

Write-Host "Done."

exit $failures
