<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke RankTurn with App.

    .DESCRIPTION
        AppPath で指定したアプリケーションを起動した状態で RankTurn を実行します。
        RankTurnOutputPath を指定した場合、ファイル出力と TeamCity 向けの(min,max,average)ログ出力を行います。
#>

param
(
    [parameter(mandatory=$true)][string]
    # 計測するアプリケーションのパス(必須)
    $AppPath,

    [array]
    # 計測するアプリケーションのオプション
    $AppOptions,

    [string]
    # アプリケーションを起動してから RankTurn が起動するまでの待機時間
    $IntervalSeconds = 0,

    [array]
    # RankTurn のオプション
    $RankTurnOptions,

    [string]
    # RankTurn の --file オプションの値
    $RankTurnOutputPath

)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\TargetManager"
Import-Module "${scriptDirectoryPath}\Modules\Runner"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Utility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

function TotalizeResult($Channel)
{
    $max = 0
    $min = [int]::MaxValue
    $sum = 0
    $sampleCount = 0

    $result = [xml](Get-Content $RankTurnOutputPath)
    foreach($campaign in $result.root.campaign)
    {
        foreach($metric in $campaign.metric)
        {
            if($metric.GetAttribute("name").Equals($Channel))
            {
                $value = $metric.GetAttribute("value")
                if($value -gt $max)
                {
                    $max = $value
                }
                if($value -lt $min)
                {
                    $min = $value
                }
                $sum += $value
                $sampleCount++
            }
        }
    }
    $average = $sum / $sampleCount
    $file = [System.IO.Path]::GetFileNameWithoutExtension($RankTurnOutputPath)
    $fileMax = $file + "Max" + "_" + $Channel
    $fileMin = $file + "Min" + "_" + $Channel
    $fileAverage = $file + "Average" + "_" + $Channel
    $fileSampleCount = $file + "SampleCount" + "_" + $Channel
    echo "##teamcity[buildStatisticValue key='$fileMax' value='$max']"
    echo "##teamcity[buildStatisticValue key='$fileMin' value='$min']"
    echo "##teamcity[buildStatisticValue key='$fileAverage' value='$average']"
    echo "##teamcity[buildStatisticValue key='$fileSampleCount' value='$sampleCount']"
}

try
{
    $RunOnTarget = "$(Get-NintendoSdkRootPath)\Tools\CommandLineTools\RunOnTarget.exe"
    $ControlTarget = "$(Get-NintendoSdkRootPath)\Tools\CommandLineTools\ControlTarget.exe"
    $RankTurnPath = "$(Get-NintendoSdkRootPath)\Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\RankTurn\Release\RankTurn.nsp"

    # RankTurn のオプションを作成
    if($RankTurnOutputPath)
    {
        $RankTurnOptions = $RankTurnOptions + "--file $RankTurnOutputPath"
    }

    echo ---RankTurn---
    echo App:$AppPath
    echo AppOptions:$AppOptions
    echo Interval:$IntervalSeconds
    echo RankTurnOptions:$RankTurnOptions
    echo --------------

    # 1.開発機リセット
    # 2.アプリケーション起動
    # 3.IntervalSeconds で指定された秒数待機
    # 4.RankTurn を起動
    # 5.開発機リセット
    Invoke-CriticalCommand "& `"$ControlTarget`" reset --hard"
    Invoke-Expression "& `"$RunOnTarget`" $AppPath --no-wait -- $AppOptions"
    Start-Sleep -Seconds $IntervalSeconds
    Invoke-Expression "& `"$RunOnTarget`" $RankTurnPath --suppress-auto-kill -- $RankTurnOptions"
    Invoke-CriticalCommand "& `"$ControlTarget`" reset --hard"

    # RankTurnOutputPath が指定されていたら min, max, average を計算して TeamCity ログとして出力する
    if($RankTurnOutputPath)
    {
        $lines = Get-Content $RankTurnOutputPath
        foreach ($line in $lines) 
        {
            echo $line
        }
        TotalizeResult("emc2mc_pm_rank_rank_turn")
        TotalizeResult("emc2mc1_pm_rank_rank_turn")
    }
}
finally
{
}

# Pass exitCode to TestRunner's result
exit $TestResult
