﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Coverity

    .DESCRIPTION
        Run coverity analysis via nact.

        Required environment variables:
        - $env:COVERITY_PATH .. Path to Coverity Analysis (e.g. "C:\Program Files\Coverity\Coverity Static Analysis\bin" )
        - $env:COVERITY_AUTH_KEY_FILE ... Authentication key for cov-commit-defects.

        Optional environment variables:
        - $env:COVERITY_CHECK_BRANCH ... Checked out branch
        - $env:COVERITY_CHECK_REVISION ... Checked out revision hash
        - $env:COVERITY_STREAM_NAME ... Stream name for cov-commit-defects
        - $env:COVERITY_CONNECT_SERVER ... Server location of Coverity connect
        - $env:COVERITY_CONNECT_PORT ... HTTP port of Coverity connect
        - $env:COVERITY_WORK_DIR ... Coverity work directory
        - $env:COVERITY_MODELING_DATA_PATH ... Coverity modeling data path
        - $env:COVERITY_AGRESSIVENESS_LEVEL ... Agressiveness level [low|medium|high]
        - $env:COVERITY_ENABLE_IMPORT_SCM ... Enable cov-import-scm [true|false]
        - $env:COVERITY_COMMIT_DEFECTS_BRANCH ... Commit defects to Coverity Connect if the analyzed branch is same as this branch.

    .INPUTS
        None
#>


$exitCode = 0

# Path settings
$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# Module settings
Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"

# NintendoSDK settings
$NintendoSdkRootPath = Get-NintendoSdkRootPath
$InvokeGit = "$scriptDirectoryPath\Invoke-Git.ps1"

# Coverity settings
$coverityBinPath     = $env:COVERITY_PATH

# Additional options for cov-build

$cov_build_additional_options = (
    "--delete-stale-tus",
    "--return-emit-failures"
)

# Additional options for cov-analyze

$cov_analyze_additional_options = (
    "--all",
    "--concurrency",
    "--security",
    "--enable-fnptr",
    "--enable-virtual",
    "--enable-exceptions",
    "--disable-fb",
    "--enable-constraint-fpp",
    "--enable-callgraph-metrics",
    "--force"
)

# Additional checkers that are used in cov-analyze

$additional_checkers = (
    "BAD_SHIFT",
    "ALLOC_FREE_MISMATCH",
    "DIVIDE_BY_ZERO",
    "ENUM_AS_BOOLEAN",
    "INCOMPATIBLE_CAST",
    "INTEGER_OVERFLOW",
    "MISSING_COMMA",
    "MIXED_ENUMS",
    "NESTING_INDENT_MISMATCH",
    "SWAPPED_ARGUMENTS"
)

$cov_build_additional_cmd_option = $cov_build_additional_options -join ' '

$cov_analyze_additional_cmd_option = $cov_analyze_additional_options -join ' '

$additional_checkers = $additional_checkers | ForEach-Object -Process {"--enable " + $_}

$cov_analyze_additional_cmd_option += ' ' + $additional_checkers -join ' '


trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}


function Parse-Params($option)
{
    $pattern = $option + "[^\s]+"
    $regex = [regex]$pattern

    $result = $regex.Matches($args) -replace $option, ""
    $result = $result -replace ",", "_"
    return $result
}

function Create-Configuration($compiler, $output)
{
    $cov_configure_command = "& `"${coverityBinPath}\cov-configure.exe`" --${compiler} --config ${output}"

    Invoke-Expression $cov_configure_command
}

# Parse parameters

$Spec = Parse-Params "specs=" @args
$BuildType = Parse-Params "BUILD=" @args
$Platform = Parse-Params "platforms=" @args

# Check parameters

if ($env:COVERITY_AUTH_KEY_FILE)
{
    $cov_commit_defects_credentials = "--auth-key-file $env:COVERITY_AUTH_KEY_FILE"
}
else
{
    throw "Auth-key file is required"
}

if ([string]::IsNullOrEmpty($Spec))
{
    throw "Spec name is required"
}

if ([string]::IsNullOrEmpty($BuildType))
{
    throw "BuildType is required"
}

if ([string]::IsNullOrEmpty($Platform))
{
    throw "Platform is required"
}

# Check existence of Coverity executables.

if ((-not(Test-Path -path "${coverityBinPath}\cov-build.exe"))`
    -or (-not(Test-Path -path "${coverityBinPath}\cov-analyze.exe"))`
    -or (-not(Test-Path -path "${coverityBinPath}\cov-configure.exe"))`
    -or (-not(Test-Path -path "${coverityBinPath}\cov-import-scm.exe"))`
    -or (-not(Test-Path -path "${coverityBinPath}\cov-commit-defects.exe")))
{
    throw "Cannot find Coverity executables under $coverityBinPath" +  "`r`nPlease set correct " + '$env:COVERITY_PATH .'
}


Write-Host "--------------------------------------------------"

Write-Host "Spec:" $Spec
Write-Host "BuildType:" $BuildType
Write-Host "Platform:" $Platform

# Setup Coverity parameters

# Environmental variable

$env:COVERITY_SUPPRESS_ASSERT = 1163

# Get Branch name

if ($env:COVERITY_CHECK_BRANCH)
{
    $branch_name = $env:COVERITY_CHECK_BRANCH
}
else
{
    $branch_name = Invoke-CriticalCommand "& $InvokeGit branch --contains"

    $branch_name = $branch_name.Replace("* ", "")
}

Write-Host "Branch name: ${branch_name}"

# Get Commit name
if ($env:COVERITY_CHECK_REVISION)
{
    $commit_name = $env:COVERITY_CHECK_REVISION
}
else
{
    $commit_name = Invoke-CriticalCommand "& $InvokeGit log head -1 | Select-String -Pattern '^commit '"

    $commit_name = "$commit_name".Substring(7, 40)
}

Write-Host "Commit hash: ${commit_name}"

# Define Stream name

if ($env:COVERITY_STREAM_NAME)
{
    $stream_name = $env:COVERITY_STREAM_NAME
}
else
{
    # Create Stream name
    $stream_name = "Siglo_default_" + $Spec + "_"  + $Platform + "_" + $BuildType
}

Write-Host "Stream name: ${stream_name}"

# Create commit message
$message = $branch_name + "_commit_" + $commit_name

Write-Host "Message: ${message}"

# Define Coverity Connect server

if ($env:COVERITY_CONNECT_SERVER)
{
    $CoverityConnectServer = $env:COVERITY_CONNECT_SERVER
}
else
{
    $CoverityConnectServer = "devcoverityhost.ncl.nintendo.co.jp"
}

# Define Coverity Connect server port

if ($env:COVERITY_CONNECT_PORT)
{
    $CoverityConnectPort = $env:COVERITY_CONNECT_PORT
}
else
{
    $CoverityConnectPort = "8090"
}

# Define work directory
# 
# If $env:COVERITY_WORK_DIR is specified, this working directory can be used after this script is finished.
# CI build step that follows this script must remove this working directory.

if ($env:COVERITY_WORK_DIR)
{
    $work_dir = $env:COVERITY_WORK_DIR
}
else
{
    $work_dir = "${NintendoSdkRootPath}\.covwork_" + $Spec + "_" + $BuildType + "_" + $commit_name.Substring(0, 7)
}

Write-Host "Work directory: ${work_dir}"

# Define Modeling data

if ($env:COVERITY_MODELING_DATA_PATH)
{
    $modeling_option = "--user-model-file ${env:COVERITY_MODELING_DATA_PATH}"
    Write-Host "Modeling data path: ${env:COVERITY_MODELING_DATA_PATH}"
}
else
{
    $modeling_option = ""
}

# Define aggressiveness-level

if ($env:COVERITY_AGRESSIVENESS_LEVEL)
{
    $aggressiveness_option = "--aggressiveness-level $env:COVERITY_AGRESSIVENESS_LEVEL"
    Write-Host "Agressiveness level: ${env:COVERITY_AGRESSIVENESS_LEVEL}"
}
else
{
    $aggressiveness_option = "--aggressiveness-level low"
    Write-Host "Agressiveness level: low (default)"
}

# Create configurations

$config_path = "${work_dir}\coverity_config_siglo.xml"

Create-Configuration "clang" $config_path

if ($LASTEXITCODE -ne 0)
{
    $exitCode = $LASTEXITCODE
    throw "cov-configure for clang has failed. exit code:$LASTEXITCODE"
}

Create-Configuration "msvc" $config_path

if ($LASTEXITCODE -ne 0)
{
    $exitCode = $LASTEXITCODE
    throw "cov-configure for msvc has failed. exit code:$LASTEXITCODE"
}

Create-Configuration "cs" $config_path

if ($LASTEXITCODE -ne 0)
{
    $exitCode = $LASTEXITCODE
    throw "cov-configure for cs has failed. exit code:$LASTEXITCODE"
}

Write-Host "Configuration file: ${config_path}"


Write-Host "--------------------------------------------------"

# Process cov-build

$cov_build_command = "& `"${coverityBinPath}\cov-build.exe`" --config $config_path --dir $work_dir $cov_build_additional_cmd_option powershell -ExecutionPolicy RemoteSigned ${scriptDirectoryPath}\Invoke-Nact.ps1 ${args}"

Write-Host "${cov_build_command}"

Invoke-Expression $cov_build_command

if ($LASTEXITCODE -ne 0)
{
    # Proceed to next step even though exit code is not zero.
    Write-Host "cov-build has failed. exit code:$LASTEXITCODE"
    $exitCode = $LASTEXITCODE
}

# Copy build-log file to artifacts directory

Copy-Item "$work_dir\build-log.txt" ".ci\Artifacts\"

# Process cov-import-scm

if ($env:COVERITY_ENABLE_IMPORT_SCM -eq "true")
{
    $cov_import_scm_command = "& `"${coverityBinPath}\cov-import-scm.exe`" --dir $work_dir --scm git"

    Invoke-Expression $cov_import_scm_command

    if ($LASTEXITCODE -ne 0)
    {
        # Proceed to next step even though exit code is not zero.
        Write-Host "cov-import-scm has failed. exit code:$LASTEXITCODE"
        $exitCode = $LASTEXITCODE
    }
}

# Process cov-analyze

$cov_analyze_command = "& `"${coverityBinPath}\cov-analyze.exe`" $aggressiveness_option $modeling_option --dir $work_dir --strip-path ${NintendoSdkRootPath} -j auto $cov_analyze_additional_cmd_option"

Write-Host "${cov_analyze_command}"

Invoke-Expression $cov_analyze_command

if ($LASTEXITCODE -ne 0)
{
    $exitCode = $LASTEXITCODE
    throw "cov-analyze has failed. exit code:$LASTEXITCODE"
}

# Copy analysis log files to artifacts directory

Copy-Item "$work_dir\output\analysis-log.txt" ".ci\Artifacts\"
Copy-Item "$work_dir\output\callgraph-metrics.csv" ".ci\Artifacts\"
Copy-Item "$work_dir\output\checked-return.csv" ".ci\Artifacts\"

# Process cov-commit-defects

if ($env:COVERITY_ENABLE_IMPORT_SCM -eq "true")
{
    $cov_commit_defects_scm_option = "--scm git"
}
else
{
    $cov_commit_defects_scm_option = ""
}

# Commit defects to Coverity Connect if $env:COVERITY_COMMIT_DEFECTS_BRANCH is specified

if (($env:COVERITY_COMMIT_DEFECTS_BRANCH) -And ($branch_name -eq $env:COVERITY_COMMIT_DEFECTS_BRANCH))
{
    $cov_commit_defects_command = "& `"${coverityBinPath}\cov-commit-defects.exe`" --dir $work_dir $cov_commit_defects_credentials --host ${CoverityConnectServer} $cov_commit_defects_scm_option --port ${CoverityConnectPort} --version ${commit_name} --target ${branch_name} --description ${message} --stream ${stream_name}"
}
else
{
    # Only output preview-report.
    $cov_commit_defects_command = "& `"${coverityBinPath}\cov-commit-defects.exe`" --dir $work_dir $cov_commit_defects_credentials --host ${CoverityConnectServer}  --port ${CoverityConnectPort} --stream ${stream_name} --preview-report-v2 $work_dir\output\preview-report.json"
}

Write-Host "${cov_commit_defects_command}"

Invoke-Expression $cov_commit_defects_command


if ($LASTEXITCODE -ne 0)
{
    $exitCode = $LASTEXITCODE
    throw "cov-commit-defects has failed. exit code:$LASTEXITCODE"
}

# Copy preview-report.json to artifacts directory

Copy-Item "$work_dir\output\preview-report.json" ".ci\Artifacts\"

# Delete working directory if $env:COVERITY_WORK_DIR is not specified.

if (-not($env:COVERITY_WORK_DIR))
{
    # Remove only if $env:COVERITY_WORK_DIR is not specified, as name of temporary work_dir will be missing after this script is finished.
    # If $env:COVERITY_WORK_DIR is specified, this working directory can be used after this script is finished.

    if (Test-Path $work_dir)
    {
        Remove-Item -path $work_dir -recurse -force
        Write-Host "Removed ${work_dir}"
    }
}

exit $exitCode
