﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Disassemble all process binaries.

    .DESCRIPTION
        Disassemble all process binaries.
#>
param
(
    [parameter(Mandatory=$True)][string] $OutputDir,
    [string] $Platform  = 'NX-NXFP2-a64',
    [string] $BuildType = 'Release',
    [string] $Revision
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"

$DisassembleObject   = "${ScriptDirectoryPath}/Disassemble-Object.ps1"

# 出力ディレクトリを作成
if (Test-Path $OutputDir)
{
    Remove-Item -Path $OutputDir -Recurse -Force
}
New-Item -Path $OutputDir -ItemType Directory -Force | Out-Null

# プロセスのバイナリが含まれる Programs のサブルート名
$programsSubRoot = @('Chris', 'Eris', 'Iris')

# Platforms と BuildType が含まれる Processes 以下の .nss を検索する
$processList = @()
foreach ($subRoot in $programsSubRoot)
{
    $processList += (Get-ChildItem -Path "$(Get-NintendoSdkRootPath)/Programs/$subRoot/Outputs/$Platform/Processes" -Include *.nss -Recurse | ?{$_ -match "\\$BuildType\\"} )
}

# プロセスが見つからなかったら終了
if (! $processList)
{
    Write-Host "Error: No process file is found. Please confirm target configuration."
    exit 1
}

# リビジョンの指定があったら出力ファイル名を .<revision>.dasm にする
if ($Revision)
{
    $fileExtension = ".$Revision.dasm"
}
else
{
    $fileExtension = ".dasm"
}

# それらに対して LOAD アトリビュートなセクションの逆アセンブルを実行
foreach ($process in $processList)
{
    Write-Host "Disassembling $process"
    $baseName = $process.BaseName
    Invoke-Expression "& `"$DisassembleObject`" -InputFile $process -OutputFile $OutputDir/$baseName$fileExtension -TargetSectionAttribute `"CODE`" -HeaderOption `"-h`""
}
