﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Build Oasis artifact

    .DESCRIPTION
        This file defines functions for building Oasis artifact
#>


param
(
    # JIRA issue key. e.g. "SIGLO-1234"
    [string]$JiraIssue,
    # Build Oasis with your own PC. Signing is disabled.
    [switch]$BuildLocally
)

function Get-TMAVersionNumber()
{
    return ((Select-String -Path .\Programs\Eris\Sources\Libraries\tmagent\Version.cpp -Pattern "const char\* TMAVersionNumber" ) -split "= "[0])[1]  -replace "`"","" -replace ";","" -replace " ",""
}

function Remove-OldInstaller()
{
    $NintendoSdkRootPath = Get-NintendoSdkRootPath
    Remove-Item $NintendoSdkRootPath\Externals\Oasis\Installer\*.msi
    Remove-Item $NintendoSdkRootPath\Externals\Oasis\Installer\*.vsix
}

function Get-HotFixString()
{
    $xml = [xml](Get-Content $OasisPath\Source\CreateOasisNXReleaseConfig.xml)
    return $xml.CreateReleaseConfiguration.HotFix
}

function Invoke-BuildSolutions()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$tmaVersion
    )

    $version = "-setversion " + $tmaVersion
    pushd $OasisPath\Source
    Invoke-CriticalCommand "..\Bin\CreateRelease.exe -config BuildOasisCreateReleaseConfig.xml $version"
    popd
}

function Invoke-CreateInstaller()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$tmaVersion
    )

    $version = "-setversion " + $tmaVersion
    pushd $OasisPath\Source
    Invoke-CriticalCommand "..\Bin\CreateRelease.exe -config BuildOasisInstallerCreateReleaseConfig.xml $version"
    popd
}

function Invoke-SignAndRenameFiles()
{
    param
    (
        $FileList
    )

    if($BuildLocally)
    {
        return
    }

    foreach($path in $FileList)
    {
        $PathExists = Test-Path $path
        If($PathExists -eq $False)
        {
            $path = Join-Path (Resolve-Path $OasisPath) $path
        }
        Write-Host "Signing $path..."
        if((Get-AuthenticodeSignature $path).Status -eq "Valid")
        {
            Write-Host "Skipping $path which is already signd."
            continue
        }
        & $SignFile -AuthenticationFile $env:SIGLO_AUTHENTICATION_FILE -InFile $path -OutFile "$path.signed"
        $PathExists = Test-Path "$path.signed"
        If($PathExists -eq $True)
        {
            Remove-Item $path
            Rename-Item -Path "$path.signed" -NewName $path
        }
    }
}

function Invoke-SignVSIXFiles()
{
    param
    (
        $VSIXFile
    )

    if($BuildLocally)
    {
        return
    }

    $path = Join-Path (Resolve-Path $OasisPath) $VSIXFile
    Write-Host "Signing assemblies in $path..."

    [string]$WorkDir = [System.IO.Path]::GetTempPath()
    $WorkDir += "VSIXTemp"

    # Unpack the VSIX - these are ZIP-readable.
    $FolderExists = Test-Path $WorkDir
    If ($FolderExists -eq $True)
    {
        Remove-Item $WorkDir -Force -Recurse
    }

    pushd $OasisPath\Source
    Invoke-CriticalCommand "..\Bin\UnpackVSIX.exe -source `"$path`" -output `"$WorkDir`""
    popd

    # Find the files we want to sign.
    $FolderExists = Test-Path $WorkDir
    If ($FolderExists -eq $True)
    {
        $FilesToSign = get-childitem $WorkDir -recurse | where {$_.extension -eq ".dll"} | Select-Object -ExpandProperty FullName
        if( $FilesToSign -ne $null )
        {
            ########################################################################
            # Sign these
            ########################################################################
            Invoke-SignAndRenameFiles -FileList $FilesToSign

            ########################################################################
            # And re-pack
            ########################################################################
            pushd $OasisPath\Source
            Invoke-CriticalCommand "..\Bin\CreateVSIX.exe -source `"$WorkDir`" -output `"$path`""
            popd
        }
    }
}

# This is called before the installer is created
function Invoke-SignFile()
{
    # Sign these VSIX-packed files.
    Invoke-SignVSIXFiles -VSIXFile "\Source\OasisVSI\NX\VS 2015\bin\x86\Release\VS 2015 Oasis NX VSI.vsix"
    Invoke-SignVSIXFiles -VSIXFile "\Source\OasisVSI\NX\VS 2017\bin\x86\Release\VS 2017 Oasis NX VSI.vsix"

    # Sign these "loose" files.
    $FileList = @(
    "\Source\Toolchain\NX\bin\NintendoTargetManager.exe",
    "\Source\Toolchain\NX\bin\NXJITHandler.exe",
    "\Source\Toolchain\NX\bin\NXSymStore.exe",
    "\Source\Toolchain\NX\lib\libnn_cdr_x64.dll",
    "\Source\Toolchain\NX\lib\libnn_tm_v140_x64.dll",
    "\Source\Toolchain\NX\lib\libnn_tm_v140_x86.dll",
    "\Source\Toolchain\NX\lib\libnn_tm_v141_x64.dll",
    "\Source\Toolchain\NX\lib\libnn_tm_v141_x86.dll",
    "\Source\Toolchain\NX\lib\Nintendo.Tm.dll",
    "\Source\OasisVSI\NX\VS 2015\bin\x86\Release\VS 2015 Oasis NX VSI.vsix"
    )

    Invoke-SignAndRenameFiles -FileList $FileList
}

# This is called after the installer is created
function Invoke-SignInstaller()
{
    param
    (
        [string]$tmaVersion
    )

    $FileList = (
    "\Source\OasisVSI\NX\VS 2017\bin\x86\Release\VS 2017 Oasis NX VSI.vsix",
    "\Source\Installer\NXSetup\Installers\NX_Visual_Studio_2015_Integration.$tmaVersion.msi",
    "\Source\Installer\NXSetup\bin\Release\ja-JP\TargetManager.$tmaVersion.msi",
    "\Source\Installer\NXSetup\OasisTMSetup\bin\Release\ja-JP\OasisTMSetup.$tmaVersion.msi"
    )

    Invoke-SignAndRenameFiles -FileList $FileList
}

function Update-Test-Artifact-Directory()
{
    param
    (
        [string]$dir
    )

    if (Test-Path "$NintendoSdkRootPath\Externals\Oasis\$dir") {
        Remove-Item -recurse -force -path "$NintendoSdkRootPath\Externals\Oasis\$dir"
    }
    Copy-Item ".\Test\TestingUtilities\$dir" "$NintendoSdkRootPath\Externals\Oasis\$dir" -recurse
}

function Copy-Artifact()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$tmaVersion,
        [string]$Hotfix
    )

    pushd $OasisPath

    $NintendoSdkRootPath = Get-NintendoSdkRootPath

    if(!$InspectorMagicOnly)
    {
        # Installer
        Copy-Item .\Source\Installer\NXSetup\bin\Release\ja-JP\TargetManager.$tmaVersion.msi                $NintendoSdkRootPath\Externals\Oasis\Installer\TargetManager.$tmaVersion$HotfixForFileName.msi
        Copy-Item .\Source\Installer\NXSetup\Installers\NX_Visual_Studio_2015_Integration.$tmaVersion.msi   $NintendoSdkRootPath\Externals\Oasis\Installer\VsiForNX_v140.$tmaVersion$HotfixForFileName.msi
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\VS 2017 Oasis NX VSI.vsix"                  "$NintendoSdkRootPath\Externals\Oasis\Installer\VsiForNX_v150.$tmaVersion$HotfixForFileName.vsix"
        # bin
        Copy-Item .\Source\Toolchain\NX\bin\NintendoTargetManager.exe   $NintendoSdkRootPath\Externals\Oasis\bin
        Copy-Item .\Source\Toolchain\NX\bin\NXJITHandler.exe            $NintendoSdkRootPath\Externals\Oasis\bin
        Copy-Item .\Source\Toolchain\NX\bin\NXSymStore.exe              $NintendoSdkRootPath\Externals\Oasis\bin
        # include
        Copy-Item .\Source\Toolchain\NX\include\cdr.h           $NintendoSdkRootPath\Externals\Oasis\include
        Copy-Item .\Source\Toolchain\NX\include\cdr_private.h   $NintendoSdkRootPath\Externals\Oasis\include
        Copy-Item .\Source\Toolchain\NX\include\tm.h            $NintendoSdkRootPath\Externals\Oasis\include
        Copy-Item .\Source\Toolchain\NX\include\tm_private.h    $NintendoSdkRootPath\Externals\Oasis\include
        # lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_cdr_x64.lib       $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_cdr_x64.dll       $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_tm_v140_x86.lib   $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_tm_v140_x64.lib   $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_tm_v141_x86.lib   $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\libnn_tm_v141_x64.lib   $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\Nintendo.Tm.dll         $NintendoSdkRootPath\Externals\Oasis\lib
        Copy-Item .\Source\Toolchain\NX\lib\Nintendo.Tm.xml         $NintendoSdkRootPath\Externals\Oasis\lib

        # pdb
        if (!(Test-Path $NintendoSdkRootPath\Externals\Oasis\pdb)) { New-Item $NintendoSdkRootPath\Externals\Oasis\pdb -itemType Directory }
        Copy-Item .\Source\TargetManager\TargetManager\NX\x64\Release\NintendoTargetManager.pdb $NintendoSdkRootPath\Externals\Oasis\pdb
        Copy-Item .\Source\Toolchain\NX\bin\NXSymStore.pdb                                      $NintendoSdkRootPath\Externals\Oasis\pdb
        Copy-Item .\Source\JITHandler\NX\bin\Release\NXJITHandler.pdb                           $NintendoSdkRootPath\Externals\Oasis\pdb
        Copy-Item .\Source\Toolchain\NX\lib\Nintendo.Tm.pdb                                     $NintendoSdkRootPath\Externals\Oasis\pdb
        Copy-Item .\Source\CoreDump\CoreDumpReport\x64\Release\libnn_cdr_x64.pdb                $NintendoSdkRootPath\Externals\Oasis\pdb

        # pdb for VSI 2015
        if (!(Test-Path $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015)) { New-Item $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015 -itemType Directory }
        Copy-Item ".\Source\OasisVSI\NX\VS 2015\bin\x86\Release\DejaTools.Localization.pdb"                         $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\OasisVSI\NX\VS 2015\bin\x86\Release\GLSLSyntaxHighlighter.pdb"                          $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\Toolchain\NX\bin\libnn_tm_v140_x86.pdb"                                                 $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\OasisVSI\NX\VS 2015\bin\x86\Release\NXDebugProject.pdb"                                 $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\OasisVSI\NX\VS 2015\bin\x86\Release\NXDMPEditor.pdb"                                    $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\DebugEngine\NX\bin\x86\Release\VS 2015 NXDebugEngine.pdb"                               $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\OasisPortProvider\NX\bin\x86\Release\VS 2015 NXPortProvider.pdb"                        $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\OasisVSI\NX\VS 2015\bin\x86\Release\VS 2015 Oasis NX VSI.pdb"                           $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015
        Copy-Item ".\Source\TargetManager\ToolbarHelper\Release\VS 2015 ToolbarHelper_x86.pdb"                      $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2015

        # pdb for VSI 2017
        if (!(Test-Path $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017)) { New-Item $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017 -itemType Directory }
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\DejaTools.Localization.pdb"                         $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\GLSLSyntaxHighlighter.pdb"                          $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\Toolchain\NX\bin\libnn_tm_v141_x86.pdb"                                                 $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\NXDebugProject.pdb"                                 $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\NXDMPEditor.pdb"                                    $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\DebugEngine\NX\bin\x86\Release\VS 2017 NXDebugEngine.pdb"                               $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisPortProvider\NX\bin\x86\Release\VS 2017 NXPortProvider.pdb"                        $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\bin\x86\Release\VS 2017 Oasis NX VSI.pdb"                           $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
        Copy-Item ".\Source\OasisVSI\NX\VS 2017\Release\VS 2017 ToolbarHelper_x86.pdb"                              $NintendoSdkRootPath\Externals\Oasis\pdb\Vsi2017
    }

    # InspectorMagic
    Copy-Item ".\Test\TestingUtilities\InspectorMagic\InspectorMagic.exe" "$NintendoSdkRootPath\Externals\Oasis\InspectorMagic\InspectorMagic.exe"
    Copy-Item ".\Test\TestingUtilities\InspectorMagic\DeterministicRandom.h" "$NintendoSdkRootPath\Externals\Oasis\InspectorMagic\DeterministicRandom.h"
    Copy-Item .\Test\TestingUtilities\InspectorMagic\NintendoSdkVcProjectSettings.local.props $NintendoSdkRootPath\Externals\Oasis\InspectorMagic\NintendoSdkVcProjectSettings.local.props
    Update-Test-Artifact-Directory InspectorMagic\TMAPIUnitTestApps
    Update-Test-Artifact-Directory InspectorMagic\TestDefinitions
    Update-Test-Artifact-Directory "InspectorMagic\TestApp"
    Update-Test-Artifact-Directory "InspectorMagic\BuildTestProjects"
    Update-Test-Artifact-Directory "InspectorMagic\DLL"
    Update-Test-Artifact-Directory "InspectorMagic\ErrorTestApps"

    # TestPilot
    Update-Test-Artifact-Directory TestPilot\bin\Release
    Update-Test-Artifact-Directory TestPilot\GoldenImages
    Copy-Item .\Test\TestingUtilities\TestPilot\AddFileToSymStore.txt $NintendoSdkRootPath\Externals\Oasis\TestPilot\AddFileToSymStore.txt
    Update-Test-Artifact-Directory TestPilotApps

    popd
}

function Invoke-UpdateNintendoSdkServiceLauncher()
{
    Invoke-CriticalCommand "& $InvokeNact BUILD=Release specs=Generic -d .\Programs\Chris\Sources\Tools\RunnerTools\NintendoSdkServiceManager"
    $targetDir = Join-Path (Get-NintendoSdkRootPath) "Externals\Oasis\Sources\Source\Toolchain\NX\bin\"
    pushd ".\Programs\Chris\Outputs\AnyCPU\Tools\RunnerTools\NintendoSdkServiceManager\NintendoSdkServiceLauncher\Release"
    Copy-Item -Force NintendoSdkServiceLauncher.exe $targetDir
    Copy-Item -Force System.Interactive.dll $targetDir
    Copy-Item -Force ToolFoundation.dll $targetDir
    popd
}

function UpdateFileForNdi()
{
    param
    (
        [string]$tmaVersion
    )
    $NintendoSdkRootPath = Get-NintendoSdkRootPath
    $tmaVersion -replace "([0-9]*).([0-9]*).([0-9]*).([0-9]*)", '$1.$2.$3+REV$4' | Out-File $NintendoSdkRootPath\Externals\Oasis\Installer\Version.txt -Encoding ASCII
}

$ErrorActionPreference = "Stop"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

#
# Import submodules
#
$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"
Import-Module "${scriptDirectoryPath}\Modules\Error"

#
# Setup Path
#
$InvokeNact = "$scriptDirectoryPath\Invoke-Nact.ps1"
$SignFile = "$scriptDirectoryPath\Sign-File.ps1"
$OasisPath = "$(Get-NintendoSdkRootPath)\Externals\Oasis\Sources"
$env:NINTENDO_SDK_ROOT = Get-NintendoSdkRootPath

##########################################################
# Entry Point
##########################################################
$tmaVersion = Get-TMAVersionNumber
[bool] $InspectorMagicOnly = ![string]::IsNullOrEmpty($env:INSPECTOR_MAGIC_ONLY)

if(!$InspectorMagicOnly)
{
    Remove-OldInstaller
}

$Hotfix = Get-HotFixString
$HotfixForFileName
if (![string]::IsNullOrEmpty($Hotfix))
{
    $HotfixForFileName = "_" + $Hotfix -replace " ",""
}

# Build
Invoke-BuildSolutions -tmaVersion $tmaVersion
Invoke-SignFile
Invoke-CreateInstaller -tmaVersion $tmaVersion
Invoke-SignInstaller -tmaVersion $tmaVersion

if ([string]::IsNullOrEmpty($Hotfix))
{
    Copy-Artifact -tmaVersion $tmaVersion
}
else
{
    Copy-Artifact -tmaVersion $tmaVersion -Hotfix $HotfixForFileName
}

if($BuildLocally)
{
    exit $LASTEXITCODE
}

UpdateFileForNdi -tmaVersion $tmaVersion

# Copy artifact to save in TeamCity
New-Item ".\Integrate\Outputs\Oasis" -ItemType Directory -ErrorAction SilentlyContinue
$CopyTarget = @(
    ".\Externals\Oasis\bin",
    ".\Externals\Oasis\include",
    ".\Externals\Oasis\InspectorMagic",
    ".\Externals\Oasis\TestPilot",
    ".\Externals\Oasis\TestPilotApps",
    ".\Externals\Oasis\Installer",
    ".\Externals\Oasis\lib"
    ".\Externals\Oasis\pdb"
    )
foreach($target in $CopyTarget)
{
    Copy-Item -Recurse $target .\Integrate\Outputs\Oasis
}

