﻿#
#    COMMAND -InputFile <inputFile> [-OutputFile <outputFile>] [-PreApiUrl <url>] [-Overwrite] [-Detail]
#            [-English] [-Detail]
#
#       -HtmlDir    <directory>       HTML があるディレクトリ。
#                                     ここ以下の html が対象となる。
#                                     省略時はカレントディレクトリとなる。
#
#       -PreApiUrl                    APIリファレンスへの位置。apiReference.js がある位置から。
#                                     "../../../Api/Html/" のように指定する。(最後の"/"も必要)
#
#       -Overwrite                    元ファイルに出力を上書きする。
#
#       -Detail                       動作の情報を詳細に表示する。省略化で、省略時は詳細を表示しない。
#
param
(
    [String]$HtmlDir = "",
    [String]$PreApiUrl = "../../../Api/Html/",
    [switch]$Overwrite = $false,
    [switch]$Detail = $false
)

$enMark = "\"  #dummy"

if ( $HtmlDir -eq "" )
{
    $HtmlDir = "." + $enMark
}


#----------------------------------------------------------------
#
function Arrange-HtmlHead( $fileName )
{
    #---- .html でない場合は戻る
    if ( -not ( [System.IO.Path]::GetExtension($fileName) -eq ".html" ) )
    {
        if ( $Detail -eq $true )
        {
            Write-Host "*** $fileName is not HTML file."
        }
        return
    }

    if ( $Detail -eq $true )
    {
        Write-Host "*** Convert $fileName ..."
    }

    # 出力する文字列
    [string[]]$dest = @()

    $isAddedScriptLine = $false
    $isSkip = $false

    #---- ファイル読み込み
    Get-Content -Encoding UTF8 ($HtmlDir + $enMark + $fileName) | foreach `
    {
        #---- BODY 閉じタグ行
        if ( $_.StartsWith( "</body>" ) )
        {
            if ( $isSkip -eq $false )
            {
                $dest += "<script>"
#                $dest += "RewriteUrl('$preApiUrl')"
                $dest += "</script>"
                $dest += "</body>"
                $isSkip = $true
            }
        }
        #---- RewriteUrl 行
        elseif ( $_.StartsWith( "RewriteUrl(" ) )
        {
            $isSkip = $true
        }
        #---- tocData行を見つけたので apiReference 行も追加する
        elseif ( $_.StartsWith( "<script type=`"text/javascript`" src=`"../tocData.js`"></script>" ) )
        {
            if ( $isAddedScriptLine -eq $false )
            {
                $isAddedScriptLine = $true
                $dest += "<script type=`"text/javascript`" src=`"../apiReference.js`"></script>"
            }
            $dest += $_
        }
        #---- apiReference 行を見つけた
        elseif ( $_.StartsWith( "<script type=`"text/javascript`" src=`"../apiReference.js`"></script>" ) )
        {
            if ( $isAddedScriptLine -eq $false )
            {
                $isAddedScriptLine = $true
                $dest += $_
            }
        }
        #---- 通常行
        else
        {
            $dest += $_
        }
    }

    #---- 上書きか別ファイルか
    if ( $Overwrite -eq $true )
    {
        $OutputFile = $HtmlDir + $enMark + $fileName
    }
    else
    {
        $OutputFile = $HtmlDir + $enMark + $fileName + ".output"
    }

    #---- 出力
    New-Item -force -ItemType file $OutputFile | Out-Null
    $dest | Out-File -Encoding UTF8 $OutputFile -Append
}

#================================================================

#---- 指定フォルダのファイルに対して処理
Get-ChildItem $HtmlDir | Foreach-Object `
{
    if ( -not $_.PSIsContainer )
    {
        Arrange-HtmlHead $_
    }

    if ( $Detail -eq $true )
    {
        Write-Host "*** Done."
    }
}

