﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nn/nn_Macro.h>
#include <nn/util/util_FormatString.h>

#include <nn/escore/estypes.h>
#include <nn/istorage/istorage.h>
#include <nn/iosc/iosc.h>
#include "cpp_c.h"

USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE

#include "es_container.h"
#include "es_storage.h"


ES_NAMESPACE_START


// Values for sanity checks
#define ES_ABSOLUTE_MAX_TICKET_SIZE     (128 * 1024 * 1024)
#define ES_ABSOLUTE_MAX_RIGHTS          (1 * 1024 * 1024)

// Buffer used for data manipulation
static u8 __buf[4 * 1024] ATTR_SHA256_ALIGN;

ESError
esVerifyTicket(IInputStream __REFVSPTR ticket, const void *certs[], u32 nCerts,
               bool doVerify, ESTicketWriter *ticketWriter,
               ESItemRightsGet *itemRightsGet, void *outTicket)
{
    ESError rv = ES_ERR_OK;
    ESV2Ticket *tp;
    IOSCPublicKeyHandle hIssuerPubKey = 0;
    u32 initialSize, remainingSize, bytes, nRead;
    u32 readPos = 0;
    u32 writePos = 0;
#if !defined(ES_READ_ONLY)
    u32 headerSize;
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash;
#endif  // ES_READ_ONLY

    // Read the v0 portion and v1 header of the ticket
    if ((rv = esSeek(ticket, 0)) != ES_ERR_OK) {
        goto end;
    }

    initialSize = sizeof(ESV2Ticket);
    if ((rv = esRead(ticket, initialSize, outTicket)) != ES_ERR_OK) {
        goto end;
    }
    readPos = initialSize;
    tp = (ESV2Ticket *) outTicket;

    // Only support RSA-2048 with SHA-256
    if (tp->sig.sigType != IOSC_SIG_RSA2048_H256) {
        esLog(ES_DEBUG_ERROR, "Unsupported signature type, %u\n", ntohl(tp->sig.sigType));
        rv = ES_ERR_INCORRECT_SIG_TYPE;
        goto end;
    }

    // Only support version 1
    if (tp->formatVersion != 2) {
        esLog(ES_DEBUG_ERROR, "Unsupported ticket version, %d\n", tp->formatVersion);
        rv = ES_ERR_INCORRECT_TICKET_VERSION;
        goto end;
    }

    // Verify the certs
    if (doVerify) {
        if ((rv = ESI_VerifyContainer(ES_CONTAINER_TKT, NULL, 0, tp->sig.sig, (const char *) tp->sig.issuer, certs, nCerts, 0, &hIssuerPubKey)) != ES_ERR_OK) {
            rv = ESI_Convert_Certificate_Error( rv );
            goto end;
        }
    }

    /*
     * Verify the ticket.  Since the ticket could potentially be large,
     * it needs to be read in by chunks
     */
    if (doVerify) {
#if !defined(ES_READ_ONLY)
        headerSize = sizeof(IOSCCertSigType) + sizeof(IOSCRsaSig2048) + sizeof(IOSCSigDummy);

        if ((rv = IOSC_GenerateHash(hashCtx, &((u8 *) tp)[headerSize], initialSize - headerSize, IOSC_SHA256_INIT, NULL)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }
#endif  // ES_READ_ONLY
    }

    if (ticketWriter) {
        if ((rv = esSeek(__REFPASSPTR(ticketWriter->ticketDest), 0)) != ES_ERR_OK) {
            goto end;
        }

        if ((rv = esWrite(__REFPASSPTR(ticketWriter->ticketDest), initialSize, tp)) != ES_ERR_OK) {
            goto end;
        }
        writePos = initialSize;
    }

    /*
     * Error check to avoid buffer overflow, but this case should
     * never happen in real life
     */
    if (sizeof(ESV2Ticket) + tp->sectTotalSize > ES_ABSOLUTE_MAX_TICKET_SIZE) {
        esLog(ES_DEBUG_ERROR, "Unexpected v2 ticket size, %u\n", sizeof(ESV2Ticket) + tp->sectTotalSize);
        rv = ES_ERR_FAIL;
        goto end;
    }

    // Hash the rest of the ticket and write it out in parallel
    remainingSize = sizeof(ESV2Ticket) + ntohl(tp->sectTotalSize) - initialSize;
    bytes = 0;

    while (bytes < remainingSize) {
        nRead = ES_MIN(remainingSize - bytes, sizeof(__buf));

        /*
         * The input ticket stream and the output ticketwriter
         * stream may be the same, so the read and write positions
         * must be maintained explicitly in this code.
         */
        if ((rv = esSeek(ticket, readPos)) != ES_ERR_OK) {
            goto end;
        }
        if ((rv = esRead(ticket, nRead, __buf)) != ES_ERR_OK) {
            goto end;
        }

        readPos += nRead;

        if (doVerify) {
#if !defined(ES_READ_ONLY)
            if ((rv = IOSC_GenerateHash(hashCtx, __buf, nRead, IOSC_SHA256_UPDATE, NULL)) != IOSC_ERROR_OK) {
                esLog(ES_DEBUG_ERROR, "Failed to generate hash - update, rv=%d\n", rv);
                rv = ESI_Translate_IOSC_Error(rv);
                goto end;
            }
#endif  // ES_READ_ONLY
        }

        if (itemRightsGet != NULL) {
            esLog(ES_DEBUG_ERROR, "itemRightsGet is not supported\n");
            rv = ES_ERR_FAIL;
            goto end;
        }

        if (ticketWriter) {
            /*
             * The input ticket stream and the output ticketwriter
             * stream may be the same, so the read and write positions
             * must be maintained explicitly in this code.
             */
            if ((rv = esSeek(__REFPASSPTR(ticketWriter->ticketDest), writePos)) != ES_ERR_OK) {
                goto end;
            }
            if ((rv = esWrite(__REFPASSPTR(ticketWriter->ticketDest), nRead, __buf)) != ES_ERR_OK) {
                goto end;
            }
            writePos += nRead;
        }

        bytes += nRead;
    }

    if (doVerify) {
#if !defined(ES_READ_ONLY)
        if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_FINAL, hash)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate hash - final, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }

        if ((rv = IOSC_VerifyPublicKeySign(hash, sizeof(hash), hIssuerPubKey, tp->sig.sig)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to verify ticket signature, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }
#endif  // ES_READ_ONLY
    }

end:
    return rv;
}


ESError
esVerifyTmdFixed(IInputStream __REFVSPTR tmd, const void *certs[], u32 nCerts,
                 ESV1TitleMeta *outTmd)
{
    ESError rv = ES_ERR_OK;
    u32 dataSize;
#if !defined(ES_READ_ONLY)
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash;
#endif  // ES_READ_ONLY

    // Read the TMD header
    if ((rv = esSeek(tmd, 0)) != ES_ERR_OK) {
        goto end;
    }

    if ((rv = esRead(tmd, sizeof(ESV1TitleMeta), outTmd)) != ES_ERR_OK) {
        goto end;
    }

    // Only support RSA-2048 with SHA-256
    if (ntohl(outTmd->sig.sigType) != IOSC_SIG_RSA2048_H256) {
        esLog(ES_DEBUG_ERROR, "Unsupported signature type, %u\n", ntohl(outTmd->sig.sigType));
        rv = ES_ERR_INCORRECT_SIG_TYPE;
        goto end;
    }

    // Only support version 1
    if (outTmd->head.version != 1) {
        esLog(ES_DEBUG_ERROR, "Unsupported TMD version, %d\n", outTmd->head.version);
        rv = ES_ERR_INCORRECT_TMD_VERSION;
        goto end;
    }

    // Verify the certs and the (TMD header + v1 TMD header hash)
    dataSize = sizeof(IOSCSigRsa2048) + sizeof(ESTitleMetaHeader) + sizeof(IOSCHash256);

    if ((rv = ESI_VerifyContainer(ES_CONTAINER_TMD, outTmd, dataSize, outTmd->sig.sig, (const char *) outTmd->sig.issuer, certs, nCerts, 0, NULL)) != ES_ERR_OK) {
        goto end;
    }

#if !defined(ES_READ_ONLY)
    // Verify the v1 TMD header using the v1 TMD header hash
    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_INIT, NULL)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, (u8 *) &outTmd->v1Head.cmdGroups[0], ES_MAX_CMD_GROUPS * sizeof(ESV1ContentMetaGroup), IOSC_SHA256_FINAL, hash)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - final, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if (memcmp(hash, outTmd->v1Head.hash, sizeof(IOSCHash256)) != 0) {
        esLog(ES_DEBUG_ERROR, "Failed to verify hash\n");
        rv = ES_ERR_VERIFICATION;
        goto end;
    }
#endif  // ES_READ_ONLY

end:
    return rv;
}


ESError
esVerifyCmdGroup(IInputStream __REFVSPTR tmd, ESV1ContentMetaGroup *groups,
                 u32 groupIndex, bool doVerify,
                 ESContentMetaSearchByIndex *cmdSearchByIndex,
                 ESContentMetaSearchById *cmdSearchById,
                 ESContentMetaGet *cmdGet)
{
    ESError rv = ES_ERR_OK;
    ESV1ContentMetaGroup *cmdGroup;
    u32 baseCmd = 0, pos, maxCmds, remainingCmds, nCmds, i, j;
    u32 curOffset = 0, nCopied = 0;
    ESV1ContentMeta *cmd;
#if !defined(ES_READ_ONLY)
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash;
#endif  // ES_READ_ONLY

    if (cmdSearchByIndex != NULL) {
        cmdSearchByIndex->nFound = 0;
    }
    if (cmdSearchById != NULL) {
        cmdSearchById->found = false;
    }

    cmdGroup = &groups[groupIndex];
    if (ntohs(cmdGroup->nCmds) == 0) {
        // Nothing to verify
        goto end;
    }

    /*
     * Read the CMDs into the global buffer for verification.  Since the
     * number of CMDs could be large, they need to be read in by chunks
     */
    for (i = 0; i < groupIndex; i++) {
        baseCmd += ntohs(groups[i].nCmds);
    }

    pos = sizeof(ESV1TitleMeta) + (baseCmd * sizeof(ESV1ContentMeta));
    if ((rv = esSeek(tmd, pos)) != ES_ERR_OK) {
        goto end;
    }

    if (doVerify) {
#if !defined(ES_READ_ONLY)
        if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_INIT, NULL)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate group hash - init, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }
#endif  // ES_READ_ONLY
    }

    remainingCmds = ntohs(cmdGroup->nCmds);
    maxCmds = sizeof(__buf) / sizeof(ESV1ContentMeta);

    while (remainingCmds > 0) {
        nCmds = ES_MIN(remainingCmds, maxCmds);

        if ((rv = esRead(tmd, nCmds * sizeof(ESV1ContentMeta), __buf)) != ES_ERR_OK) {
            goto end;
        }

        if (doVerify) {
#if !defined(ES_READ_ONLY)
            if ((rv = IOSC_GenerateHash(hashCtx, __buf, nCmds * sizeof(ESV1ContentMeta), IOSC_SHA256_UPDATE, NULL)) != IOSC_ERROR_OK) {
                esLog(ES_DEBUG_ERROR, "Failed to generate group hash - update, rv=%d\n", rv);
                rv = ESI_Translate_IOSC_Error(rv);
                goto end;
            }
#endif  // ES_READ_ONLY
        }

        // Search for the requested CMD
        cmd = (ESV1ContentMeta *) __buf;
        for (i = 0; i < nCmds; i++) {
            if (cmdSearchByIndex != NULL) {
                for (j = 0; j < cmdSearchByIndex->nIndexes; j++) {
                    if (cmdSearchByIndex->indexes[j] == ntohs(cmd->index)) {
                        cmdSearchByIndex->outInfos[j].id = ntohl(cmd->cid);
                        cmdSearchByIndex->outInfos[j].index = ntohs(cmd->index);
                        cmdSearchByIndex->outInfos[j].type = ntohs(cmd->type);
                        cmdSearchByIndex->outInfos[j].size = ntohll(cmd->size);

                        cmdSearchByIndex->nFound++;
                        break;
                    }
                }
            }

            if (cmdSearchById != NULL) {
                if (cmdSearchById->id == ntohl(cmd->cid)) {
                    *cmdSearchById->outCmd = *cmd;
                    cmdSearchById->found = true;
                }
            }

            if (cmdGet != NULL) {
                if (curOffset >= cmdGet->offset && curOffset < (cmdGet->offset + cmdGet->nInfos)) {
                    cmdGet->outInfos[nCopied].id = ntohl(cmd->cid);
                    cmdGet->outInfos[nCopied].index = ntohs(cmd->index);
                    cmdGet->outInfos[nCopied].type = ntohs(cmd->type);
                    cmdGet->outInfos[nCopied].size = ntohll(cmd->size);

                    nCopied++;
                }
            }

            cmd++;
            curOffset++;
        }

        remainingCmds -= nCmds;
    }

    if (doVerify) {
#if !defined(ES_READ_ONLY)
        if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_FINAL, hash)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate group hash - final, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }

        if (memcmp(hash, cmdGroup->groupHash, sizeof(IOSCHash256)) != 0) {
            esLog(ES_DEBUG_ERROR, "Failed to verify group hash\n");
            rv = ES_ERR_VERIFICATION;
            goto end;
        }
#endif  // ES_READ_ONLY
    }

end:
    return rv;
}

ESError
esVerifyTmdContent(IInputStream __REFVSPTR tmd, ESContentId contentId,
           ESV1TitleMeta *inTmd, ESV1ContentMeta *outCmd)
{
    ESError rv = ES_ERR_OK;
    u32 i;
    ESContentMetaSearchById cmdSearchById, *cmdSearchByIdPtr = NULL;
    bool found = false;

    // Verify each individual CMD group
    for (i = 0; i < ES_MAX_CMD_GROUPS; i++) {
        if (outCmd) {
            cmdSearchById.id = contentId;
            cmdSearchById.outCmd = outCmd;

            cmdSearchByIdPtr = &cmdSearchById;
        }

        if ((rv = esVerifyCmdGroup(tmd, inTmd->v1Head.cmdGroups, i, true, NULL, cmdSearchByIdPtr, NULL)) != ES_ERR_OK) {
            goto end;
        }

        if (cmdSearchByIdPtr && cmdSearchByIdPtr->found) {
            found = true;
        }
    }

    if (outCmd != NULL && !found) {
        esLog(ES_DEBUG_ERROR, "Failed to find content ID\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

end:
    return rv;

}

ESError
esVerifyTmd(IInputStream __REFVSPTR tmd, const void *certs[], u32 nCerts,
            ESContentId contentId, ESV1TitleMeta *outTmd,
            ESV1ContentMeta *outCmd)
{
    ESError rv = ES_ERR_OK;

    if ((rv = esVerifyTmdFixed(tmd, certs, nCerts, outTmd)) != ES_ERR_OK) {
        goto end;
    }

    rv = esVerifyTmdContent(tmd, contentId, outTmd, outCmd);

end:
    return rv;
}


#if !defined(ES_READ_ONLY)
static ESError
__signHelper(ESTitleId signerTitleId, IOSCHash256 hash,
             IOSCEccSig *outSig, IOSCEccEccCert *outCert)
{
    ESError rv = ES_ERR_OK;
    IOSCSecretKeyHandle hSignKey = 0;
    IOSCCertName certName;
    u32 titleIdHigh, titleIdLow;

    // Generate a private key and ephemeral cert
    if ((rv = IOSC_CreateObject(&hSignKey, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to create object for secret key, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_GenerateKey(hSignKey)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate secret key, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    titleIdHigh = (u32) (signerTitleId >> 32);
    titleIdLow = (u32) signerTitleId;
    memset(certName, 0, sizeof(IOSCCertName));
    nn::util::SNPrintf((char *) certName, sizeof(ES_APP_CERT_PREFIX) + 16, "%s%08x%08x", ES_APP_CERT_PREFIX, titleIdHigh, titleIdLow);

    if ((rv = IOSC_GenerateCertificate(hSignKey, certName, (IOSCEccSignedCert *) outCert)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate cert, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    // Sign with the private key
    if ((rv = IOSC_GeneratePublicKeySign(hash, sizeof(IOSCHash256), hSignKey, (u8 *) outSig)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to sign, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

end:
    if (hSignKey != 0) {
        (void) IOSC_DeleteObject(hSignKey);
    }

    return rv;
}


ESError
esSign(ESTitleId signerTitleId, IInputStream __REFVSPTR data, u32 dataSize,
       void *outSig, u32 outSigSize, void *outCert, u32 outCertSize)
{
    ESError rv = ES_ERR_OK;
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash;
    u32 bytes, remainingSize;

    if (dataSize == 0 || outSig == NULL || outSigSize != ES_SIGNATURE_SIZE || outCert == NULL || outCertSize != ES_SIGNING_CERT_SIZE || sizeof(IOSCEccSig) != ES_SIGNATURE_SIZE || sizeof(IOSCEccSignedCert) != ES_SIGNING_CERT_SIZE) {
        esLog(ES_DEBUG_ERROR, "Invalid arguments\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    // Compute data hash
    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_INIT, NULL)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = esSeek(data, 0)) != ES_ERR_OK) {
        goto end;
    }

    remainingSize = dataSize;
    while (remainingSize > 0) {
        bytes = ES_MIN(remainingSize, sizeof(__buf));

        if ((rv = esRead(data, bytes, __buf)) != ES_ERR_OK) {
            goto end;
        }

        if ((rv = IOSC_GenerateHash(hashCtx, __buf, bytes, IOSC_SHA256_UPDATE, NULL)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate hash - update, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }

        remainingSize -= bytes;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_FINAL, hash)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - final, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = __signHelper(signerTitleId, hash, (IOSCEccSig *) outSig, (IOSCEccEccCert *) outCert)) != ES_ERR_OK) {
        goto end;
    }

end:
    return rv;
}


ESError
esSignData(ESTitleId signerTitleId, const void *data, u32 dataSize,
           IOSCEccSig *outSig, IOSCEccEccCert *outCert)
{
    ESError rv = ES_ERR_OK;
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash;

    if (dataSize == 0 || outSig == NULL || outCert == NULL) {
        esLog(ES_DEBUG_ERROR, "Invalid arguments\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    // Compute data hash
    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_SHA256_INIT, NULL)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, (u8 *) data, dataSize, IOSC_SHA256_FINAL, hash)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - update, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = __signHelper(signerTitleId, hash, outSig, outCert)) != ES_ERR_OK) {
        goto end;
    }

end:
    return rv;
}


ESError
esVerifyDeviceSign(ESTitleId signerTitleId,
                   IInputStream __REFVSPTR data, u32 dataSize,
                   const IOSCEccSig *sig, ESSigType sigType,
                   const IOSCEccEccCert *signingCert,
                   IOSCPublicKeyHandle hDevicePubKey)
{
    ESError rv = ES_ERR_OK;
    IOSCPublicKeyHandle hSigningPubKey = 0;
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    IOSCHash256 hash256;
    IOSCHash hash;
    IOSCCertSigType certSigType;
    u8 *hashPtr;
    u32 hashSize, hashType, bytes, remainingSize, hSize, dSize;

    (void) signerTitleId;

    if (dataSize == 0 || sig == NULL || (sigType != ES_SIG_TYPE_ECC_SHA1 && sigType != ES_SIG_TYPE_ECC_SHA256) || signingCert == NULL || hDevicePubKey == 0) {
        esLog(ES_LOG_ERROR, "Invalid arguments\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    // Verify signing cert
    certSigType = ntohl(signingCert->sig.sigType);
    if (certSigType == IOSC_SIG_ECC_H256) {
        hashPtr = hash256;
        hashSize = sizeof(hash256);
        hashType = IOSC_HASH_SHA256;
    } else if (certSigType == IOSC_SIG_ECC) {
        hashPtr = hash;
        hashSize = sizeof(hash);
        hashType = IOSC_HASH_SHA1;
    } else {
        esLog(ES_LOG_ERROR, "Invalid signing cert sig type %u\n", certSigType);
        rv = ES_ERR_INVALID;
        goto end;
    }

    hSize = sizeof(IOSCCertSigType) + sizeof(IOSCEccSig) + sizeof(IOSCEccPublicPad) + sizeof(IOSCSigDummy);
    dSize = sizeof(IOSCEccEccCert) - hSize;

    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, ((u8 *) signingCert) + hSize, dSize, IOSC_HASH_LAST | hashType, hashPtr)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - final, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_VerifyPublicKeySign(hashPtr, hashSize, hDevicePubKey, (u8 *) signingCert->sig.sig)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to verify signingCert, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    // Compute data hash
    if (sigType == ES_SIG_TYPE_ECC_SHA256) {
        hashPtr = hash256;
        hashSize = sizeof(hash256);
        hashType = IOSC_HASH_SHA256;
    } else if (sigType == ES_SIG_TYPE_ECC_SHA1) {
        hashPtr = hash;
        hashSize = sizeof(hash);
        hashType = IOSC_HASH_SHA1;
    } else {
        esLog(ES_LOG_ERROR, "Invalid data sig type %u\n", sigType);
        rv = ES_ERR_INVALID;
        goto end;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - init, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = esSeek(data, 0)) != ES_ERR_OK) {
        goto end;
    }

    remainingSize = dataSize;
    while (remainingSize > 0) {
        bytes = ES_MIN(remainingSize, sizeof(__buf));

        if ((rv = esRead(data, bytes, __buf)) != ES_ERR_OK) {
            goto end;
        }

        if ((rv = IOSC_GenerateHash(hashCtx, __buf, bytes, IOSC_HASH_MIDDLE | hashType, NULL)) != IOSC_ERROR_OK) {
            esLog(ES_DEBUG_ERROR, "Failed to generate hash - update, rv=%d\n", rv);
            rv = ESI_Translate_IOSC_Error(rv);
            goto end;
        }

        remainingSize -= bytes;
    }

    if ((rv = IOSC_GenerateHash(hashCtx, NULL, 0, IOSC_HASH_LAST | hashType, hashPtr)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to generate hash - final, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    // Verify signature
    if ((rv = IOSC_CreateObject(&hSigningPubKey, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to create object for signingPubKey, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_ImportPublicKey((u8 *) signingCert->pubKey, NULL, hSigningPubKey)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to import public key for signingPubKey, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = IOSC_VerifyPublicKeySign(hashPtr, hashSize, hSigningPubKey, (u8 *) sig)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to verify, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

end:
    if (hSigningPubKey) {
        (void) IOSC_DeleteObject(hSigningPubKey);
    }

    return rv;
}


ESError
esVerifySign(ESTitleId signerTitleId,
             IInputStream __REFVSPTR data, u32 dataSize,
             const void *sig, u32 sigSize, ESSigType sigType,
             const void *certs[], u32 nCerts)
{
    ESError rv = ES_ERR_OK;
    IOSCCertName certName;
    IOSCEccEccCert *signingCert, *deviceCert;
    IOSCPublicKeyHandle hDevicePubKey = 0;
    char *deviceId, *msId;
    u32 signingCertSize, deviceCertSize, titleIdHigh, titleIdLow;

    if (dataSize == 0 || sig == NULL || sigSize != ES_SIGNATURE_SIZE || (sigType != ES_SIG_TYPE_ECC_SHA1 && sigType != ES_SIG_TYPE_ECC_SHA256) || certs == NULL || nCerts == 0 || sizeof(IOSCEccSig) != ES_SIGNATURE_SIZE) {
        esLog(ES_DEBUG_ERROR, "Invalid arguments\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    // Find signing and device certs
    titleIdHigh = (u32) (signerTitleId >> 32);
    titleIdLow = (u32) signerTitleId;
    memset(certName, 0, sizeof(IOSCCertName));
    nn::util::SNPrintf((char *) certName, sizeof(ES_APP_CERT_PREFIX) + 16, "%s%08x%08x", ES_APP_CERT_PREFIX, titleIdHigh, titleIdLow);

    if ((rv = ESI_FindCert((char *) certName, 0, certs, nCerts, (void **) &signingCert, &signingCertSize, &deviceId)) != ES_ERR_OK) {
        goto end;
    }

    if ((rv = ESI_FindCert(deviceId, 1, certs, nCerts, (void **) &deviceCert, &deviceCertSize, &msId)) != ES_ERR_OK) {
        goto end;
    }

    // Verify device cert
    if ((rv = IOSC_CreateObject(&hDevicePubKey, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE)) != IOSC_ERROR_OK) {
        esLog(ES_DEBUG_ERROR, "Failed to create object for public key, rv=%d\n", rv);
        rv = ESI_Translate_IOSC_Error(rv);
        goto end;
    }

    if ((rv = ESI_VerifyContainer(ES_CONTAINER_DEV, deviceCert, deviceCertSize, deviceCert->sig.sig, (const char *) deviceCert->sig.issuer, certs, nCerts, hDevicePubKey, NULL)) != ES_ERR_OK) {
        goto end;
    }

    if ((rv = esVerifyDeviceSign(signerTitleId, data, dataSize, (const IOSCEccSig *) sig, sigType, (const IOSCEccEccCert *) signingCert, hDevicePubKey)) != ES_ERR_OK) {
        goto end;
    }

end:
    if (hDevicePubKey) {
        (void) IOSC_DeleteObject(hDevicePubKey);
    }

    return rv;
}


#define ES_MAX_CERTS    32  /* need an arbitrary limit */

ESError
esVerifyCerts(const void *certs[], u32 nCerts,
              const void *verifyChain[], u32 nVerifyCerts)
{
    ESError rv = ES_ERR_OK;
    u32 i, j;
    u8 isLeaf[ES_MAX_CERTS];
    void *leafCerts[ES_MAX_CERTS];
    u32 nLeafCerts;
    void *chainCerts[ES_MAX_CERTS];
    u32 nChainCerts;
    u8 *subject;
    u8 *issuer;
    IOSCName fullName;  /* fully qualified "Issuer-Subject" name */

    if (nCerts + nVerifyCerts > ES_MAX_CERTS) {
        rv = ES_ERR_INVALID;
        goto end;
    }

    /*
     * These cases would fall through and return rv = ES_ERR_OK, but better to
     * fail and indicate that nothing useful happened.
     */
    if (nCerts == 0 || nCerts + nVerifyCerts == 0) {
        rv = ES_ERR_INVALID;
        goto end;
    }

    /*
     * If a verify chain is supplied, the processing is simpler.
     */
    if (verifyChain == NULL || nVerifyCerts == 0) {
        goto emptyChainCase;
    }

    for (i = 0; i < nCerts; i++) {
        rv = ESI_VerifyCert(certs[i], verifyChain, nVerifyCerts);
        if (rv != ES_ERR_OK) {
            goto end;
        }
    }

    rv = ES_ERR_OK;
    goto end;

emptyChainCase:
    /*
     * Handle the case of starting from scratch with an empty cert store.
     *
     * This is functionally equivalent to a topological sort on the certs using
     * 'x' is issuer of 'y' as the relationship and then finding the nodes with
     * no outgoing edges.  Note that this solution conserves memory and stack at
     * the expense of N**2 running time, but N is limited to ES_MAX_CERTS.
     *
     * No assumption is made about the order of the certs in the input lists.
     */
    for (i = 0; i < nCerts; i++) {
        rv = ESI_GetCertNames(certs[i], &issuer, &subject);
        if (rv != ES_ERR_OK) {
            goto end;
        }

        /*
         * It is not safe to handle a self-signed certificate, unless the public key is already
         * known by the client code.  Anyone with OpenSSL can manufacture a consistent self-signed
         * certificate with whatever issuer and subject names are required.  Since this code
         * needs the root public keys to be baked in, there is no real point in dealing with root
         * certificates (the iGware root certs are not distributed for this reason).
         */
        if (strncmp((const char *) issuer, (const char *) subject, sizeof(IOSCName)) == 0) {
            esLog(ES_DEBUG_ERROR, "Self-signed certificate (root cert) rejected: %s\n", subject);
            rv = ES_ERR_VERIFICATION;
            goto end;
        }

        /*
         * For Nintendo certs, there is an underscore instead of a dash between the root and
         * the next level CA.
         * XXX this is pretty cheesy:  depends on having Nintendo as the start of both strings.
         */
        const char *separator = "-";
        if (strncmp((const char *) issuer, "Nintendo", 8) == 0 &&
            strncmp((const char *) subject, "Nintendo", 8) == 0) {
            separator = "_";
        }
        nn::util::SNPrintf((char *) fullName, sizeof(fullName), "%s%s%s",
            (const char *) issuer, (const char *) separator, (const char *) subject);
        for (j = 0; j < nCerts; j++) {
            if (j == i) continue;
            rv = ESI_GetCertNames(certs[j], &issuer, &subject);
            if (rv != ES_ERR_OK) {
                goto end;
            }
            if (strncmp((const char *) fullName, (const char *) issuer, sizeof(fullName)) == 0) {
                /*
                 * certs[i] is issuer of certs[j]
                 */
                isLeaf[i] = 0;
                break;
            }
        }
        if (j == nCerts) {
            isLeaf[i] = 1;
        }
    }

    nLeafCerts = 0;
    nChainCerts = 0;
    for (i = 0; i < nCerts; i++) {
        if (isLeaf[i]) leafCerts[nLeafCerts++] = (void *) certs[i];
        else chainCerts[nChainCerts++] = (void *) certs[i];
    }

    for (i = 0; i < nLeafCerts; i++) {
        rv = ESI_VerifyCert(leafCerts[i], (const void **) chainCerts, nChainCerts);
        if (rv != ES_ERR_OK) {
            goto end;
        }
    }

end:
    return rv;
}
#endif  // ES_READ_ONLY

ES_NAMESPACE_END
