﻿// 文字コード:UTF-8
/// @file
#include <lib/debug/Print.hpp>

#include <limits>
#include <string>
#include <wchar.h>
#if 0
#include <lib/Axis3.hpp>
#include <lib/Vector3.hpp>
#endif
#include <lib/logger/Logger.hpp>
#include <lib/Unused.hpp>

//------------------------------------------------------------------------------
namespace lib {
namespace debug {

//------------------------------------------------------------------------------
::lib::logger::Logger& Print::Logger()
{
    // static変数としておくことで、未初期化状態での参照を防ぐ。
    static ::lib::logger::Logger logger(::lib::logger::LogLevel::Min);
    return logger;
}

//------------------------------------------------------------------------------
void Print::PrintFloatDetail(const float aValue)
{
    LIB_UNUSED(aValue);
#if defined(LIB_OPT_USE_PRINTF)
    const uint32_t valueU32 = *reinterpret_cast<const uint32_t*>(&aValue);
    const uint32_t sign = (valueU32 >> 31) & 0x1;
    const uint32_t exponent = (valueU32 >> 23) & 0xff;
    const uint32_t mantissa = (valueU32) & 0x7fffff;
    LIB_PRINTF("float=%f, ", aValue);
    LIB_PRINTF("hex=0x%08x, ", valueU32);
    LIB_PRINTF("sign=%u, ", sign);
    LIB_PRINTF("exponent=%u, ", exponent);
    LIB_PRINTF("mantissa=%u, ", mantissa);
    // 式での表示。Rubyで有効な計算式での表記。
    if (exponent == 0xff) {
        // exponentが0xff(255)の時。
        // これは無限大もしくはNaNとなるので、expressionは表示しない。
        LIB_PRINTF("\n");
    } else if (exponent == 0) {
        // 非正規化数の時。
        // 0は本来非正規化数ではないが、このカテゴリに入れている。
        LIB_PRINTF("expression=%c(%u / 8388608.0) * (2 ** -126)\n",
            ((sign == 0) ? '+' : '-'), mantissa);
    } else {
        // 一般の時。
        LIB_PRINTF("expression=%c(1 + %u / 8388608.0) * (2 ** %u)\n",
            ((sign == 0) ? '+' : '-'), mantissa, exponent - 127);
    }
#endif
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintfImpl(const char* aFormat, ...)
{
    va_list list;
    va_start(list, aFormat);

    const ::lib::ShortString result = ::lib::ShortString::FromFormatVA(
        list,
        aFormat
        );
    va_end(list);
    return result; //lint !e438
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const ::lib::ShortString& aVar)
{
    return HelSPrintfImpl("%s", aVar.readPtr());
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const char aVar)
{
    return HelSPrintfImpl("%c", aVar);
}
//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const int8_t aVar)
{
    return HelSPrintfImpl("%hhd", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const int16_t aVar)
{
    return HelSPrintfImpl("%hd", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const int aVar)
{
    return HelSPrintfImpl("%d", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const int64_t aVar)
{
    return HelSPrintfImpl("%lld", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const uint8_t aVar)
{
    return HelSPrintfImpl("%hhu", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const uint16_t aVar)
{
    return HelSPrintfImpl("%hu", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const uint32_t aVar)
{
    return HelSPrintfImpl("%u", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const uint64_t aVar)
{
    return HelSPrintfImpl("%llu", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const float aVar)
{
    return HelSPrintfImpl("%f", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const double aVar)
{
    return HelSPrintfImpl("%lf", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const bool aVar)
{
    return HelSPrintfImpl("%s", aVar ? "true" : "false");
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(void* aVar)
{
    return HelSPrintfImpl("%p", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const void* aVar)
{
    return HelSPrintfImpl("%p", aVar);
}

//------------------------------------------------------------------------------
::lib::ShortString Print::HelSPrintVarImpl(const char* aVar)
{
    return HelSPrintfImpl("%s", aVar);
}

}} // namescape
// EOF
