﻿// 文字コード:UTF-8
/// @file
#pragma once

//------------------------------------------------------------------------------
namespace lib {
/// コンパイル時定数で処理を分けたい時に用いる便利クラス。
/// @details
/// 通常下記の例のようにSFINAEと関数オーバーロードの仕組みを使って、
/// 型による処理を分けることを実現する。
///
/// これはTBoolがtrueの時のみTypeという型が定義される。
/// このTypeを参照するようなコードを書くことで、SFINAEによって
/// Typeが存在する場合つまりTBoolがtrueの場合のみ、Typeを参照する方の
/// コードが用いられる。
///
/// C++11ではstd:enable_if (in type_traits)として定義されているのと同様のもの。
///
/// テンプレート第二引数のTResultTypeはstdのenable_ifと同様の定義にするために
/// 用意しているが、あまり使い道は分かっていない。通常、デフォルト引数の
/// voidのままでよいと思われる。
/// * 使い方
/// @code
/// // ある型Typeがintに変換できるなら特定の実装を用い、それ以外なら
/// // 汎用の実装を用いる。
/// template <class TType> void func(TType aValue,
///     typename ::lib::TypeTraitsEnableIf<
///         ::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type* = nullptr)
/// {
///     // TTypeがintに変換「できる」型ならこちらの関数が呼ばれる
/// }
///
/// template <class TType> void func(TType aValue,
///     typename ::lib::TypeTraitsEnableIf<
///         !::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type* = nullptr)
/// {
///     // TTypeがintに変換「できない」型ならこちらの関数が呼ばれる
/// }
/// @endcode
template<bool TBool, typename TResultType = void> struct TypeTraitsEnableIf
{
};

// TBoolがtrueの時のテンプレート特殊化。
template<typename TResultType> struct TypeTraitsEnableIf<true, TResultType>
{
    typedef TResultType Type;
};

} // namespace
// EOF
