﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <cstdarg>
#include <cctype>

LIB_FORWARD_DECLARE_2(lib, class U16String);

namespace lib {

/// 特別な事をしない限りはアロケーションが発生しない文字列
class String
{
public:
    static void UnitTest();
    /// 容量1, 長さ0の空文字を作成
    static String GetEmpty();
    /// UTF16からの変換で文字列を生成
    static String FromUtf16(const char16_t* aStr);
    static String FromUtf16(const U16String& aStr);
    /// 整形済の文字列を生成
    static String FromFormat(int aCap, const char* aStr, ...);
    static String FromFormatV(int aCap, const char* aStr, ::std::va_list aVaList);
    static String FromFormatAsPossible(int aCap, const char* aStr, ...);
    static String FromFormatVAsPossible(int aCap, const char* aStr, ::std::va_list aVaList);
    /// 容量ゼロの状態で生成
    String();
    /// 容量を指定して生成
    explicit String(int aCap);
    /// C文字列から生成
    explicit String(const char* aStr);
    /// 容量を指定して生成し、C文字列を設定
    explicit String(int cap, const char* aStr);
    /// 文字列から生成
    String(const String& aStr);
    /// 破棄
    ~String();
    /// 有効か？
    bool isValid() const { return mCap > 0; }
    /// 容量
    int capacity() const { return mCap; }
    /// C文字列として取得
    const char* buffer() const { SYS_ASSERT(isValid()); return mBuf; }
    /// 文字の取得
    char& at(int i) { SYS_ASSERT(0 <= i && i < mCap); SYS_ASSERT_POINTER(mBuf); return mBuf[i]; }
    /// 文字の取得
    char at(int i) const { SYS_ASSERT(0 <= i && i < mCap); SYS_ASSERT_POINTER(mBuf); return mBuf[i]; }
    // 空文字列化する、メモリは開放しない
    void clear();
    // メモリを開放
    void reset();
    /// 容量の再設定、メモリも再確保される
    void reset(int aCap);
    /// 文字列の再設定、容量が再計算されメモリも再確保される
    void reset(const char* aStr);
    /// 文字列の再設定、容量が再設定されメモリも再確保される
    void reset(const String& aStr);
    /// 容量を再設定し、文字列を設定、メモリも再確保される
    void reset(int aCap, const char* aStr);
    /// 空文字列か？
    bool isEmpty() const;
    /// 同一か？
    bool operator==(const char* aStr) const;
    bool operator==(const String& aStr) const { return *this == aStr.buffer(); }
    /// 同一か？
    bool equals(const char* aStr, bool aWillIgnoreCase = false) const;
    /// 指定文字列で始まるか？
    bool startsWith(const char* aStr, bool aWillIgnoresCase = false) const;
    /// 指定文字列で終わるか？
    bool endsWith(const char* aStr, bool aWillIgnoresCase = false) const;
    /// 指定文字列の開始位置(インデックス)を取得、見つからなければ-1を返す、startIdxで検索開始インデックスを指定
    int getIndexOf(const char* aStr, int aStartIdx = 0, bool aWillIgnoresCase = false) const;
    /// 指定文字の位置(インデックス)を取得、見つからなければ-1を返す、startIdxで検索開始インデックスを指定
    int getIndexOf(char aChar, int aStartIdx = 0, bool aWillIgnoresCase = false) const;
    /// 指定文字列の終了位置(インデックス)を取得、見つからなければ-1を返す、startIdxで検索開始インデックスを指定
    int getLastIndexOf(const char* aStr, int aStartIdx = -1, bool aWillIgnoresCase = false) const;
    /// 部分文字列の取得、末尾まで
    String getSubstring(int aStartIdx) const { return getSubstringByLength(aStartIdx, getLength() - aStartIdx); }
    /// 部分文字列の取得、長さ指定
    String getSubstringByLength(int aStartIdx, int aLen) const;
    /// 部分文字列の取得、指定
    String getSubstringByIndex(int aStartIdx, int aEndIdx) const { return getSubstringByLength(aStartIdx, aEndIdx - aStartIdx); }
    /// 設定(欠落不許可)
    void set(const String& aStr);
    /// 設定(欠落不許可)
    String& operator=(const String& aStr);
    /// 設定(欠落許容)
    void setAsPossible(const String& aStr);
    /// C文字列から設定(欠落不許可)
    void set(const char* aStr);
    /// C文字列から設定(欠落不許可)
    String& operator=(const char* aStr);
    /// C文字列から設定(欠落許容)
    void setAsPossible(const char* aStr);
    /// U16文字列から設定(欠落不許可)
    void setFromUtf16(const char16_t* aStr);
    /// U16文字列から設定(欠落不許可)
    void setFromUtf16(const U16String& aStr);
    /// U16文字列から設定(欠落許可)
    void setFromUtf16AsPossible(const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void setFromUtf16AsPossible(const uint16_t* aStr) { setFromUtf16AsPossible(reinterpret_cast<const char16_t*>(aStr)); }
#endif
    /// U16文字列から設定(欠落許可)
    void setFromUtf16AsPossible(const U16String& aStr);
    /// 限界長指定、「aLen」以降はカット、capacityは変わらない
    void limit(int aLen);
    /// 文字列の置き換え。removeStr -> insertStr(変換後のcapacityオーバーは不許可)
    void replaceAll(const char* aRemoveStr, const char* aInsertStr);
    /// 文字の置き換え。removeWord -> insertWord
    void replaceWord(char removeWord, char insertWord);
    /// 大文字を小文字に変換した文字列を取得
    String toLowerCase() const;
    /// UTF8として正しいか？
    bool isCorrectUtf8() const;
    /// UTF8として正しくなるよう末尾を補正する
    void correctTailAsUtf8IfPossible();
    /// 文字列長の取得
    int getLength() const;
    /// 指定文字を含むか？
    bool contains(char aCh) const;
    /// 指定文字列を含むか？
    bool contains(const char* aStr) const;
    /// ASCII文字だけで構成されているか？（マルチバイト文字を含んでいないか？）
    bool containsOnlyAscii() const;
    /// 先頭と末尾の空白文字を削除
    void strip();
    /// 文字の連結
    String& operator+=(const String& aStr);
    /// 文字の連結
    String operator+(const String& aStr) const;
    /// 整形(欠落不許可)
    void format(const char* aStr, ...);
    /// 整形(欠落不許可)
    void formatV(const char* aStr, ::std::va_list aVaList);
    /// 整形(欠落許容)
    void formatAsPossible(const char* aStr, ...);
    /// 整形(欠落許容)
    void formatVAsPossible(const char* aStr, ::std::va_list aVaList);

private:
    static bool Equals(const char aLhs, const char aRhs, bool aWillIgnoresCase);

    char* mBuf;
    int mCap;
};

//------------------------------------------------------------------------------
inline bool String::Equals(const char aLhs, const char aRhs, bool aWillIgnoresCase)
{
    if (aWillIgnoresCase) {
        return (::std::tolower(aLhs) == ::std::tolower(aRhs));
    } else {
        return (aLhs == aRhs);
    }
}

//------------------------------------------------------------------------------
} // namespace
// EOF
