﻿// 文字コード:UTF-8
/// @file
/// @author nakano
#pragma once

#include <lib/Type.hpp>
#include <lib/debug/Assert.hpp>
#include <lib/Unused.hpp>

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB
//@{
/// ポインタのラッパークラス。
template <typename TPointee>
class Pointer
{
private:
    typedef TPointee* Pointer::*SafeBool; ///< SafeBoolイディオムのための型。

public:
    /// @name コンストラクタ
    //@{
    /// オブジェクトが設定されていない状態で初期化。
    Pointer() : mPtr(nullptr) {}
    /// 設定するオブジェクトを指定して初期化。
    Pointer(TPointee* aObj) : mPtr(aObj) {}
    //@}

    /// @name ポインタの設定
    //@{
    /// オブジェクトを設定解除する。
    void reset() { mPtr = nullptr; }

    /// オブジェクトを再設定する。既に設定されてろうがされてなかろうが呼べる。
    void reset(TPointee* aObj) { mPtr = aObj; }

    /// オブジェクトを設定する。既に設定されてない場合のみ呼べる。
    inline void set(TPointee* aObj);

    /// オブジェクトを交換する。既に設定されている場合のみ呼べる。
    inline void change(TPointee* aObj);
    //@}

    /// @name 設定済みのオブジェクトを設定解除する
    //@{
    void unset() { unset(&ref()); }
    inline void unset(TPointee* aObj);
    //@}

    /// @name 設定済みか
    //@{
    bool isValid() const { return mPtr != nullptr; }
    //@}

    /// @name 設定されていないか
    //@{
    bool isEmpty() const { return mPtr == nullptr; }
    bool isNull() const { return isEmpty(); }
    //@}

    /// @name オブジェクトのアクセス
    //@{
    inline TPointee& ref() const;
    //@}

    /// @name ポインタに生アクセスする
    //@{
    /// @note
    /// ・これを使うことでコードが綺麗になる場合以外は
    /// ref()を使うようにしてください。
    /// （Nullポインタのエラーチェックがされるため）
    TPointee* get() const { return mPtr; }
    //@}

    /// @name operator演算子の実装
    //@{
    TPointee* operator->() const { return &ref(); }
    TPointee& operator*() const { return ref(); }
    /// ポインタがnullでなければtrue、ポインタがnullならfalseを返す。
    /// @details
    /// 「::lib::ScopedPtr」と同等。
    inline operator SafeBool() const;
    //@}

private:
    TPointee* mPtr;
};
//@}

//------------------------------------------------------------------------------
template <typename TPointee>
void Pointer<TPointee>::set(TPointee* aObj)
{
    SYS_ASSERT(!isValid());
    mPtr = aObj;
}

//------------------------------------------------------------------------------
template <typename TPointee>
void Pointer<TPointee>::change(TPointee* aObj)
{
    SYS_ASSERT(isValid());
    mPtr = aObj;
}

//------------------------------------------------------------------------------
template <typename TPointee>
void Pointer<TPointee>::unset(TPointee* aObj)
{
    SYS_ASSERT(isValid());
    SYS_ASSERT(mPtr == aObj);
    LIB_UNUSED(aObj);
    mPtr = nullptr;
}

//------------------------------------------------------------------------------
template <typename TPointee>
TPointee& Pointer<TPointee>::ref() const
{
    SYS_ASSERT(isValid());
    return *mPtr;
}

//------------------------------------------------------------------------------
template <class TPointee>
Pointer<TPointee>::operator SafeBool() const
{
    return mPtr == nullptr ? nullptr : &Pointer::mPtr;
}

} // namespace
// EOF
