﻿// 文字コード:UTF-8
/// @file
#include "lib/JsonReader.hpp"

#include <cstdio>
#include <cstring>
#include <algorithm>
#include "lib/StringParser.hpp"

namespace lib {

//------------------------------------------------------------------------------
JsonReader::JsonReader(const char* aStr, int aStrSize)
: mBuf(aStr)
, mBufSize(aStrSize)
, mBufIndex()
{
    readUnnecessaryCharsAndComment();
}

//------------------------------------------------------------------------------
JsonReader::~JsonReader()
{
}

//------------------------------------------------------------------------------
bool JsonReader::readObjectOpening()
{
    bool isOk = readSpecifiedChar('{');
    readUnnecessaryCharsAndComment();
    return isOk;
}

//------------------------------------------------------------------------------
bool JsonReader::readObjectClosing()
{
    bool isOk = readSpecifiedChar('}');
    readUnnecessaryCharsAndComment();
    return isOk;
}

//------------------------------------------------------------------------------
bool JsonReader::readArrayOpening()
{
    bool isOk = readSpecifiedChar('[');
    readUnnecessaryCharsAndComment();
    return isOk;
}

//------------------------------------------------------------------------------
bool JsonReader::readArrayClosing()
{
    bool isOk = readSpecifiedChar(']');
    readUnnecessaryCharsAndComment();
    return isOk;
}

//------------------------------------------------------------------------------
bool JsonReader::readKey(char* aStr, int aStrCap)
{
    auto startIndex = mBufIndex;
    if (readString(aStr, aStrCap) || readIdentifier(aStr, aStrCap)) {
        if (readSpecifiedChar(':')) {
            readUnnecessaryCharsAndComment();
            return true;
        }
    }
    mBufIndex = startIndex;
    return false;
}

//------------------------------------------------------------------------------
bool JsonReader::readInt(int* aValue)
{
    if (mBufIndex >= mBufSize) {
        return false;
    }
    int startIndex = mBufIndex;
    char c = mBuf[mBufIndex];
    // 符号
    if (c == '-') {
        mBufIndex++;
    }
    // 読み進める
    while (1) {
        if (mBufIndex >= mBufSize) {
            return false;
        }
        c = mBuf[mBufIndex];
        if (c >= '0' && c <= '9') {
            mBufIndex++;
            continue;
        }
        if (c == '.') {
            // 小数点と思われる場合はインデックスを巻き戻して失敗させる
            mBufIndex = startIndex;
            return false;
        }
        break;
    }
    // 1文字も読めなかった
    if (startIndex == mBufIndex) {
        return false;
    }
    // 数値文字列を取得する
    {
        char str[256];
        int strCount = mBufIndex - startIndex;
        ::std::memcpy(str, &mBuf[startIndex], ::std::min(static_cast<int>(sizeof(str)), strCount));
        // 終端を付加する
        str[(strCount < sizeof(str)) ? strCount : sizeof(str) - 1] = 0;
        *aValue = StringParser::ParseDecInt(str);
    }
    readUnnecessaryCharsAndComment();
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readFloat(float* aValue)
{
    if (mBufIndex >= mBufSize) {
        return false;
    }
    int startIndex = mBufIndex;
    char c = mBuf[mBufIndex];
    // 符号
    if (c == '-') {
        mBufIndex++;
    }
    // 読み進める
    while (1) {
        if (mBufIndex >= mBufSize) {
            return false;
        }
        c = mBuf[mBufIndex];
        if (
            (c >= '0' && c <= '9') ||
            c == '.'
        ) {
            mBufIndex++;
            continue;
        }
        break;
    }
    // 1文字も読めなかった
    if (startIndex == mBufIndex) {
        return false;
    }
    // 数値文字列を取得する
    {
        char str[256];
        int strCount = mBufIndex - startIndex;
        ::std::memcpy(str, &mBuf[startIndex], ::std::min(static_cast<int>(sizeof(str)), strCount));
        // 終端を付加する
        str[(strCount < sizeof(str)) ? strCount : sizeof(str) - 1] = 0;
        *aValue = StringParser::ParseFloat(str);
    }
    readUnnecessaryCharsAndComment();
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readBool(bool* aValue)
{
    if (readSpecifiedString("true")) {
        *aValue = true;
        readUnnecessaryCharsAndComment();
        return true;
    }
    else if (readSpecifiedString("false")) {
        *aValue = false;
        readUnnecessaryCharsAndComment();
        return true;
    }
    else {
        return false;
    }
}

//------------------------------------------------------------------------------
bool JsonReader::readString(char* aStr, int aStrCap)
{
    if (mBufIndex >= mBufSize) {
        return false;
    }
    char c = 0;
    // 1文字目の条件「"」を満たしていれば1文字目を読む
    {
        c = mBuf[mBufIndex];
        if (c == '"') {
            mBufIndex++;
        } else {
            return false;
        }
    }
    int startIndex = mBufIndex;
    int endIndex  = startIndex;
    // 2文字目以降を「"」まで読んでいく
    while (1) {
        if (mBufIndex >= mBufSize) {
            return false;
        }
        c = mBuf[mBufIndex];
        // 終端の「"」まで来た場合
        if (c == '"') {
            endIndex = mBufIndex;
            mBufIndex++;
            break;
        }
        // 「\"」は読み飛ばす
        if (readSpecifiedString("\\\"")) {
            continue;
        }
        mBufIndex++;
    }
    // 文字列の取り出し、「\"」を除去する必要があるので単純なコピーではできない
    {
        int strCount = 0;
        int strCountSrc = endIndex - startIndex;
        for (int i = 0; i < strCountSrc; ++i) {
            if (strCount >= aStrCap) {
                break;
            }
            if (
                i < strCountSrc - 1 &&
                mBuf[startIndex + i + 0] == '\\' &&
                mBuf[startIndex + i + 1] == '"'
            ) {
                // 「\」だけコピーせずに読み飛ばす
                continue;
            }
            aStr[strCount++] = mBuf[startIndex + i];
        }
        // 終端を付加する
        aStr[(strCount < aStrCap) ? strCount : aStrCap - 1] = 0;
    }
    readUnnecessaryCharsAndComment();
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readValue(const char** aStartPos, const char** aEndPos)
{
    auto startPos = currentPosition();
    auto dummyBool = bool();
    auto dummyInt = int();
    auto dummyFloat = float();
    char dummyString[1] = {};
    if (readBool(&dummyBool) ||
        readInt(&dummyInt) ||
        readFloat(&dummyFloat) ||
        readString(dummyString, 1) ||
        readObject() ||
        readArray())
    {
        if (aStartPos) {
            *aStartPos = startPos;
        }
        if (aEndPos) {
            *aEndPos = currentPosition();
        }
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
bool JsonReader::readIdentifier(char* aStr, int aStrCap)
{
    char c = 0;
    int startIndex = mBufIndex;
    // 読んでいく
    while (1) {
        if (mBufIndex >= mBufSize) {
            return false;
        }
        c = mBuf[mBufIndex];
        if (
            (c >= 'A' && c <= 'Z') ||
            (c >= 'a' && c <= 'z') ||
            // 最初の文字以外は数字も許可する
            ((startIndex != mBufIndex) && c >= '0' && c <= '9') ||
            c == '_'
        ) {
            mBufIndex++;
            continue;
        }
        break;
    }
    // 1文字も読めなかった
    if (startIndex == mBufIndex) {
        return false;
    }
    // キーを設定する
    {
        int strCount = mBufIndex - startIndex;
        ::std::memcpy(aStr, &mBuf[startIndex], ::std::min(aStrCap, strCount));
        // 終端を付加する
        aStr[(strCount < aStrCap) ? strCount : aStrCap - 1] = 0;
    }
    readUnnecessaryCharsAndComment();
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readObject()
{
    auto startIndex = mBufIndex;
    // オブジェクトの開始位置
    if (!readObjectOpening()) {
        return false;
    }
    // オブジェクト要素を飛ばす
    while (1) {
        // キー
        char dummy[1] = {};
        if (!readKey(dummy, sizeof(dummy))) {
            break;
        }
        // 値
        if (!readValue()) {
            mBufIndex = startIndex;
            return false;
        }
    }
    // オブジェクトの終了位置
    if (!readObjectClosing()) {
        mBufIndex = startIndex;
        return false;
    }
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readArray()
{
    auto startIndex = mBufIndex;
    // 配列の開始位置
    if (!readArrayOpening()) {
        return false;
    }
    // 配列要素を飛ばす
    while (readValue());
    // 配列の終了位置
    if (!readArrayClosing()) {
        mBufIndex = startIndex;
        return false;
    }
    return true;
}

//------------------------------------------------------------------------------
void JsonReader::readUnnecessaryCharsAndComment()
{
    while (1) {
        if (mBufIndex >= mBufSize) {
            return;
        }
        char c = 0;
        c = mBuf[mBufIndex];
        // 不要文字を読み飛ばす
        if (
            mBufIndex < mBufSize &&
            (
                c == ' ' ||
                c == '\t' ||
                c == '\r' ||
                c == '\n' ||
                c == ',' ||
                c == ';'
                )
        ) {
            mBufIndex++;
            continue;
        }
        // コメント「//」を行末まで読み飛ばす
        if (readSpecifiedString("//")) {
            while (1) {
                if (mBufIndex >= mBufSize) {
                    return;
                }
                c = mBuf[mBufIndex];
                if (c == '\r' || c == '\n') {
                    break;
                }
                mBufIndex++;
            }
            continue;
        }
        break;
    }
}

//------------------------------------------------------------------------------
bool JsonReader::readSpecifiedChar(const char aChar)
{
    if (mBufIndex >= mBufSize) {
        return false;
    }
    if (mBuf[mBufIndex] != aChar) {
        return false;
    }
    mBufIndex++;
    return true;
}

//------------------------------------------------------------------------------
bool JsonReader::readSpecifiedString(const char* aStr)
{
    int strLen = static_cast<int>(::std::strlen(aStr));
    if (mBufIndex + strLen > mBufSize) {
        return false;
    }
    for (int i = 0; i < strLen; ++i) {
        char c = mBuf[mBufIndex + i];
        if (c != aStr[i]) {
            return false;
        }
    }
    mBufIndex += strLen;
    return true;
}

} // namespace
// EOF
