﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/IntrusiveListNode.hpp"

//------------------------------------------------------------------------------
namespace lib {

/// 侵入リスト、リスト化されるものの内部にリストにされるための構造「IntrusiveListNode」を持つ
template<class TType>
class IntrusiveList
{
public:
    template <typename TOwner, typename TNode>
    class IteratorImpl
    {
    public:
        IteratorImpl(TNode* aNode) : mNode(aNode) {}
        TOwner& operator*() { SYS_ASSERT_POINTER(mNode); return mNode->owner(); }
        void operator++() { mNode = mNode ? mNode->nextPtr() : nullptr; }
        bool operator!=(IteratorImpl& aIterator) { return (mNode != aIterator.mNode); }
    private:
        TNode* mNode;
    };

    using This = IntrusiveList<TType>;
    using Node = IntrusiveListNode<TType>;
    using Iterator      = IteratorImpl<TType, Node>;
    using ConstIterator = IteratorImpl<const TType, const Node>;

    IntrusiveList() : mTerminal(this) {};
    ~IntrusiveList() { SYS_ASSERT(isEmpty()); }
    Node& first() { SYS_ASSERT(!isEmpty()); return mTerminal.next(); }
    const Node& first() const { return const_cast<This*>(this)->first(); }
    Node* firstPtr() { return (isEmpty() ? 0 : mTerminal.nextPtr()); }
    const Node* firstPtr() const { return (isEmpty() ? 0 : mTerminal.nextPtr()); }
    Node& last() { SYS_ASSERT(!isEmpty()); return mTerminal.previous(); }
    const Node& last() const { return const_cast<This*>(this)->last(); }
    Node* lastPtr() { return (isEmpty() ? 0 : mTerminal.previousPtr()); }
    const Node* lastPtr() const { return (isEmpty() ? 0 : mTerminal.previousPtr()); }
    Node& terminal() { return mTerminal; }
    bool isEmpty() const { return !mTerminal.hasNext(); }
    int getCount() const;
    int getIndexOf(const Node& aNode) const;
    Node& getAt(int aIndex);
    const Node& getAt(int aIndex) const { return ((This*)this)->getAt(aIndex); }
    void append(Node* aNode)  { SYS_ASSERT_POINTER(aNode); mTerminal.insertBeforeSelf(aNode); }
    void prepend(Node* aNode) { SYS_ASSERT_POINTER(aNode); mTerminal.insertAfterSelf(aNode); }

    Iterator      begin()       { return {firstPtr()}; }
    ConstIterator begin() const { return {firstPtr()}; }
    Iterator      end()       { return {nullptr}; }
    ConstIterator end() const { return {nullptr}; }

private:
    Node mTerminal;
};

//------------------------------------------------------------------------------
template<class TType>
int IntrusiveList<TType>::getCount() const
{
    int size = 0;
    for (const Node* n = firstPtr(); n; n = n->nextPtr()) {
        size++;
    }
    return size;
}

//------------------------------------------------------------------------------
template<class TType>
int IntrusiveList<TType>::getIndexOf(const Node& aNode) const
{
    int index = 0;
    for (const Node* n = firstPtr(); n; n = n->nextPtr()) {
        if (n == &aNode) {
            return index;
        }
        index++;
    }
    SYS_ASSERT_NOT_REACHED();
    return 0;
}

//------------------------------------------------------------------------------
template<class TType>
typename IntrusiveList<TType>::Node& IntrusiveList<TType>::getAt(int aIndex)
{
    int i = 0;
    for (Node* n = firstPtr(); n; n = n->nextPtr()) {
        if (i == aIndex) {
            return *n;
        }
        ++i;
    }
    SYS_ASSERT_NOT_REACHED();
    return *(Node*)0;
}

} // namespace
// EOF
