﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/priv.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/ucred.h>
#include <sys/resourcevar.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/refcount.h>
#include <siglo/client.h>

struct ucred *crhold(struct ucred *cr)
{
    if (cr != NULL) {
        refcount_acquire(&(cr->cr_ref));
    }
    return cr;
}

void crfree(struct ucred *cr)
{
    if (cr != NULL &&
        refcount_release(&(cr->cr_ref)) == 1 &&
        cr->cr_container != NULL)
    {
        client_free(cr->cr_container);
    }
    return;
}

int securelevel_gt(struct ucred *cr, int level)
{
    return 0;
}

/*
 * Determine whether the subject represented by cred can "see" a socket.
 * Returns: 0 for permitted, ENOENT otherwise.
 */
int cr_canseeinpcb(struct ucred *cred, struct inpcb *inp)
{
    return 0;
}

/*
 * Determine if u1 "can see" the subject specified by u2.
 * Returns: 0 for permitted, an errno value otherwise
 * Locks: none
 * References: *u1 and *u2 must not change during the call
 *             u1 may equal u2, in which case only one reference is required
 */
int cr_cansee(struct ucred *u1, struct ucred *u2)
{
    return 0;
}

/*
 * Determine whether the subject represented by cred can "see" a socket.
 * Returns: 0 for permitted, ENOENT otherwise.
 */
int cr_canseesocket(struct ucred *cred, struct socket *so)
{
    return 0;
}

/*
 * Fill in a struct xucred based on a struct ucred.
 */
void cru2x(struct ucred *cr, struct xucred *xcr)
{
    return;
}

int priv_check_cred(struct ucred *cred, int priv, int flags)
{
    // Is this an internal caller?
    if(cred == NULL){
        return 0;
    }

    if(cred->cr_groups[0] == 0)
    {
        return 0;
    }

    // Privileges restricted by newer and current library versions
    switch (priv)
    {
        // Blocked privileges
    case PRIV_NET_ROUTE:
    case PRIV_NET_BPF:
    case PRIV_NETINET_RESERVEDPORT:
    case PRIV_SO_DEBUG:
    case PRIV_SO_ACCEPTFILTER:
    case PRIV_SO_SETFIB:
    case PRIV_SO_LABEL:
    case PRIV_SO_PEERLABEL:
    case PRIV_SO_USER_COOKIE:
    case PRIV_IP_RSVP_ON:
    case PRIV_IP_RSVP_OFF:
    case PRIV_IP_RSVP_VIF_ON:
    case PRIV_IP_RSVP_VIF_OFF:
    case PRIV_IP_HDRINCL:
    case PRIV_IP_FAITH:
    case PRIV_IP_PORTRANGE:
    case PRIV_TCP_MD5SIG:
    case PRIV_SOCK_SEQPACKET:
    case PRIV_NET_SETLLADDR:
    case PRIV_NET_ADDIFADDR:
    case PRIV_NET_DELIFADDR:
    case PRIV_NET_SETIFFLAGS:
    case PRIV_NET_SETIFMTU:
    case PRIV_NET_SETIFMETRIC:
    case PRIV_NETINET_CARP:
    case PRIV_NETINET_MROUTE:
    case PRIV_RAW_NON_ICMP:
    case PRIV_SIOC:
    case PRIV_DUPLICATE_SOCKET:
        printf("Operation Not Permitted. Need Privilege ID: %d\n", priv);
        return EPERM;
    default:
        // not restricted yet
        break;
    }

    // Privileges which were restricted by previous library versions
    struct client* container;
    if((container = (struct client*)cred->cr_container) != NULL){
        if(lim_cur(&container->cl_proc, RLIMIT_LIB_COMPATIBILITY) <= ((rlim_t)3)){
            switch (priv)
            {
                // Blocked privileges
            case PRIV_IP_MULTICAST_IF:
            case PRIV_IP_MULTICAST_TTL:
            case PRIV_IP_MULTICAST_LOOP:
            case PRIV_IP_ADD_MEMBERSHIP:
            case PRIV_IP_DROP_MEMBERSHIP:
            case PRIV_IP_MULTICAST_VIF:
                printf("Operation Not Permitted. Need Privilege ID: %d\n", priv);
                return EPERM;
            default:
                // not restricted yet
                break;
            }
        }
    }

    // access granted
    return 0;
}

int priv_check(struct thread *td, int priv)
{
    // Is this an internal caller?
    if(td == NULL){
        return 0;
    }
    return priv_check_cred(td->td_ucred, priv, 0);
}

}

