﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/mbuf.h>

/* malloc region definitions */
MALLOC_DEFINE(M_DEVBUF,  "devbuf",   "Misc device buffers");
MALLOC_DEFINE(M_IP6OPT,  "ip6opt",   "IPv6 options");
MALLOC_DEFINE(M_IP6NDP,  "ip6ndp",   "IPv6 Neighbor Discovery");
MALLOC_DEFINE(M_NETUSER, "inetuser", "net process memory");

/* main memory pool */
extern size_t   mempool_size;
extern uint8_t* mempool;

/* network related limits */
static unsigned long  maxmbufmem;   /* limits mbuf memory */
int nmbufs;                         /* limits number of mbufs */
int nmbclusters;                    /* limits number of mbuf clusters */
int nmbjumbop;                      /* limits number of page size jumbo clusters */
int nmbjumbo9;                      /* limits number of 9k jumbo clusters */
int nmbjumbo16;                     /* limits number of 16k jumbo clusters */

/* thread yield */
void kern_yield(int prio)
{
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
}

/* notify of arrival of a new ifnet interface */
void devctl_notify(
    const char *__system, const char *__subsystem,
    const char *__type, const char *__data)
{
#if BSDDEBUG
    printf("[net_init]: %s %s %s device\n", __type, __subsystem, __system);
#endif
    return;
}

/* device driver register callback */
void module_register_init(const void *arg)
{
    const moduledata_t *mod = (const moduledata_t*)arg;
#if BSDDEBUG
    printf("[net_init]: %s\n", mod->name);
#endif
    mod->evhand(NULL, MOD_LOAD, mod->priv);
    return;
}

void *osd_get(u_int type, struct osd *osd, u_int slot)
{
    STUB_WARNING();
    return NULL;
}

void kassert_halt(const char* function, int line)
{
    printf("assert at %s:%d\n", function, line);
    NN_ABORT_UNLESS(false);
}

void sched_bind(struct thread *td, int cpu)
{
    STUB_WARNING();
}

void sched_unbind(struct thread *td)
{
    STUB_WARNING();
}

static void tunable_mbinit(void *dummy)
{

#if MALLOC_COLLECT_STATS
    // malloc header is stored in front of each chunk cutting
    // available memory for page aligned allocations
    // in half
    maxmbufmem = (mempool_size / 4 * 3) / 2;
#else
    maxmbufmem = mempool_size / 4 * 3;
#endif

    TUNABLE_ULONG_FETCH("kern.ipc.maxmbufmem", &maxmbufmem);
    TUNABLE_INT_FETCH("kern.ipc.nmbclusters",  &nmbclusters);
    TUNABLE_INT_FETCH("kern.ipc.nmbjumbop",    &nmbjumbop);
    TUNABLE_INT_FETCH("kern.ipc.nmbjumbo9",    &nmbjumbo9);
    TUNABLE_INT_FETCH("kern.ipc.nmbjumbo16",   &nmbjumbo16);
    TUNABLE_INT_FETCH("kern.ipc.nmbufs",       &nmbufs);

    if (maxmbufmem == 0 || maxmbufmem > mempool_size / 4 * 3)
    {
        maxmbufmem = mempool_size / 4 * 3;
    }
    if (nmbclusters == 0)
    {
        nmbclusters = maxmbufmem / MCLBYTES / 4;
    }
    if (nmbjumbop == 0)
    {
        nmbjumbop = maxmbufmem / MJUMPAGESIZE / 4;
    }
    if (nmbjumbo9 == 0)
    {
        nmbjumbo9 = maxmbufmem / MJUM9BYTES / 6;
    }
    if (nmbjumbo16 == 0)
    {
        nmbjumbo16 = maxmbufmem / MJUM16BYTES / 6;
    }
    if (nmbufs < nmbclusters + nmbjumbop + nmbjumbo9 + nmbjumbo16)
    {
        nmbufs = lmax(
                maxmbufmem / MSIZE / 5,
                nmbclusters + nmbjumbop + nmbjumbo9 + nmbjumbo16
        );
    }
}

SYSINIT(tunable_mbinit, SI_SUB_KMEM, SI_ORDER_MIDDLE, tunable_mbinit, NULL);

}
