﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SystemThreadDefinition.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/bus.h>
#include <sys/interrupt.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <siglo/thread.h>

static const int SWI_STACK_SIZE = 8192;
static  MALLOC_DEFINE(M_ITHREAD, "ithread", "Interrupt Threads");

struct swi_p
{
    uint8_t           stack[SWI_STACK_SIZE]
                            __attribute__((aligned(nn::os::StackRegionAlignment)));
    nn::os::EventType event;
    NetworkThread     tid;
    volatile int      terminate;
    driver_intr_t*    handler;
    void*             arg;
};

static void swi_processor(void* arg)
{
    struct swi_p* p = (struct swi_p*)arg;
    while (!p->terminate) {
        nn::os::WaitEvent(&p->event);
        p->handler(p->arg);
    }
    return;
}

int swi_add_ex(driver_intr_t handler, void *arg, int pri, int cpu, void **cookiep)
{
    struct swi_p* p =
        (struct swi_p*)memalign(
                nn::os::StackRegionAlignment,
                sizeof(struct swi_p),
                M_ITHREAD,
                M_WAITOK | M_ZERO);

    if (p != NULL) {
        p->handler   = handler;
        p->arg       = arg;
        p->terminate = 0;
        nn::os::InitializeEvent(
                &p->event,
                false,
                nn::os::EventClearMode_AutoClear
        );
        NN_ABORT_UNLESS(
            CreateNetworkThread(
                &(p->tid),
                swi_processor,
                (void*)p,
                p->stack,
                sizeof(p->stack),
                NN_SYSTEM_THREAD_PRIORITY(socket, CoreStackSwi),
                cpu) == 0
        );
        SetNetworkThreadNamePointer(&(p->tid), NN_SYSTEM_THREAD_NAME(socket, CoreStackSwi));
        StartNetworkThread(&p->tid);
        *cookiep = (void*)p;
        return 0;
    }
    return EINVAL;
}

void swi_sched(void *cookie, int flags)
{
    struct swi_p* p = (struct swi_p*)cookie;
    nn::os::SignalEvent(&(p->event));
    return;
}

}

