﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/condvar.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>

static MALLOC_DEFINE(M_COND, "icondition", "condition variable storage");

int
cv_wait_sig_l(struct cv *cvp, struct lock_object *lock)
{
    nn::os::ConditionVariableType* cond = (nn::os::ConditionVariableType*)cvp->siglo_ptr;
    nn::os::MutexType* siglo_mtx = (nn::os::MutexType*)(((struct mtx*)lock)->siglo_ptr);
    nn::os::WaitConditionVariable(cond, siglo_mtx);
    return 0;
}

void
cv_broadcastpri(struct cv *cvp, int pri)
{
    nn::os::ConditionVariableType* cond = (nn::os::ConditionVariableType*)cvp->siglo_ptr;
    nn::os::BroadcastConditionVariable(cond);
}

void
cv_init(struct cv *cvp, const char *desc)
{
    nn::os::ConditionVariableType* cond;
    cond = (nn::os::ConditionVariableType*)malloc(sizeof(nn::os::ConditionVariableType), M_COND, M_ZERO | M_WAITOK);
    NN_ABORT_UNLESS(cond != NULL);
    nn::os::InitializeConditionVariable(cond);
    cvp->siglo_ptr = (void*)cond;
}

void
cv_destroy(struct cv *cvp)
{
    nn::os::ConditionVariableType* cond = (nn::os::ConditionVariableType*)cvp->siglo_ptr;
    nn::os::FinalizeConditionVariable(cond);
    free(cond, M_COND);
}

int
cv_timedwait_sig_sbt_l(struct cv *cvp, struct lock_object *lock, sbintime_t sbt, sbintime_t pr, int flags)
{
    /* C_ABSOLUTE flag is ignored:
     * used only in poll and select calls which were modified
     * to use relatvie time
     */

    nn::os::ConditionVariableType* cond = (nn::os::ConditionVariableType*)cvp->siglo_ptr;
    nn::os::MutexType* siglo_mtx = (nn::os::MutexType*)(((struct mtx*)lock)->siglo_ptr);
    nn::os::ConditionVariableStatus cv_status;

    cv_status = nn::os::TimedWaitConditionVariable(cond,
                                                   siglo_mtx,
                                                   nn::TimeSpan::FromMicroSeconds(sbt_to_microseconds(sbt)));
    return (cv_status == nn::os::ConditionVariableStatus_Timeout) ? EWOULDBLOCK : 0;
}

}

