﻿/*-
 * Copyright (c) 1982, 1988, 1991, 1993
 *    The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *    @(#)systm.h    8.7 (Berkeley) 3/29/95
 * $FreeBSD$
 */

#ifndef _SYS_SYSTM_H_
#define _SYS_SYSTM_H_

#include <stdio.h>
#include <string.h>

#ifdef _KERNEL

__BEGIN_DECLS

#include <siglo/file_scope_mutex.h>
#include <machine/atomic.h>
#include <machine/param.h>
#include <sys/callout.h>
#include <sys/libkern.h>
#include <sys/proc.h>

#ifdef SMP
#error SMP enabled
#endif

/*
 * Enable to print stubbed/not implemented functions
 */
#define PRINT_STUBBED_FUNCTIONS 0
#if PRINT_STUBBED_FUNCTIONS
#define STUB_WARNING()      printf("stub func: %s\n", __FUNCTION__)
#else
#define STUB_WARNING()      (void)(0)
#endif

/* nonzero to print verbose messages */
extern int bootverbose;

/* this definition is replaced with siglo_printf by preprocessor */
int    printf(const char *, ...);

/*
 * General function declarations.
 */

#define fubyte(addr) (*((uint8_t*)(addr)))

struct inpcb;
struct malloc_type;
struct mtx;
struct socket;
struct ucred;

/*
 * Hash
 */
void    hashdestroy(void *, struct malloc_type *, u_long);
void    *hashinit(int count, struct malloc_type *type, u_long *hashmask);
void    *hashinit_flags(int count, struct malloc_type *type,
        u_long *hashmask, int flags);
#define HASH_NOWAIT    0x00000001
#define HASH_WAITOK    0x00000002

/*
 * Missing in libc
 */
void    log(int, const char *, ...);
int     kvprintf(char const *, void (*)(int, void*), void *, int, __va_list);

/*
 * A bit different than standard one
 */
int     snprintf(char*, size_t, const char*, ...);

/*
 * Data copy
 */
int     copyinstr(const void * __restrict udaddr, void * __restrict kaddr,
        size_t len, size_t * __restrict lencopied)
        __nonnull(1) __nonnull(2);
int     copyin(const void * __restrict udaddr, void * __restrict kaddr,
        size_t len) __nonnull(1) __nonnull(2);
int     copyout(const void * __restrict kaddr, void * __restrict udaddr,
        size_t len) __nonnull(1) __nonnull(2);
int     copyout_nofault(const void * __restrict kaddr, void * __restrict udaddr,
            size_t len) __nonnull(1) __nonnull(2);

/*
 * Credentials
 */
int     cr_cansee(struct ucred *u1, struct ucred *u2);
int     cr_canseesocket(struct ucred *cred, struct socket *so);
int     cr_canseeinpcb(struct ucred *cred, struct inpcb *inp);

/*
 * Tunables
 */
char    *getenv(const char *name);
int     getenv_ulong(const char *name, unsigned long *data);
int     getenv_int(const char *name, int *data);
int     getenv_string(const char *name, char *data, int size);

/* Timeouts */
typedef void timeout_t(void *);	/* timeout function type */

/*
 * Sleep related
 */
int     _sleep(void *chan, struct lock_object *lock, int pri, const char *wmesg,
        sbintime_t sbt, sbintime_t pr, int flags) __nonnull(1);
#define msleep(chan, mtx, pri, wmesg, timo)                                     \
        _sleep((chan), &(mtx)->lock_object, (pri), (wmesg),                     \
        tick_sbt * (timo), 0, C_HARDCLOCK)
#define msleep_sbt(chan, mtx, pri, wmesg, bt, pr, flags)                        \
         _sleep((chan), &(mtx)->lock_object, (pri), (wmesg), (bt), (pr),        \
        (flags))
#define tsleep(chan, pri, wmesg, timo)                                          \
        _sleep((chan), NULL, (pri), (wmesg), tick_sbt * (timo),                 \
        0, C_HARDCLOCK)
void    wakeup(void *chan) __nonnull(1);
void    wakeup_one(void *chan) __nonnull(1);
void    DELAY(int usec);

/*
 * Scheduler related
 */
void    sched_bind(struct thread *td, int cpu);
void    sched_unbind(struct thread *td);
#define sched_pin()     (void)(0)
#define sched_unpin()   (void)(0)
#define intr_event_bind(e, i) (0)
void    kern_yield(int);
#define cpu_spinwait()  kern_yield(0)

/*
 * Kernel mem allocator
 */
void*   _sbrk_r(ssize_t len);

/*
 * Unit number allocation API. (kern/subr_unit.c)
 */
struct unrhdr;
struct unrhdr *new_unrhdr(int low, int high, struct mtx *mutex);
void   delete_unrhdr(struct unrhdr *uh);
int    alloc_unr(struct unrhdr *uh);
int    alloc_unr_specific(struct unrhdr *uh, u_int item);
void   free_unr(struct unrhdr *uh, u_int item);
void   kassert_halt(const char* function, int line) __attribute__((__noreturn__));

#define panic(...)                                                      \
         do {                                                           \
            printf(__VA_ARGS__);                                        \
            printf("\n");                                               \
            kassert_halt(__FUNCTION__, __LINE__);                       \
        } while(0)

#define abort()            kassert_halt(__FUNCTION__, __LINE__)
#define kassert_panic(...) kassert_halt(__FUNCTION__, __LINE__)

/* This option enables additional run time checks,
 * normally disabled for performance reasons
 */
#ifdef  INVARIANTS
#define KASSERT(exp,msg)                                                \
        do {                                                            \
            if (__predict_false(!(exp)))                                \
                kassert_panic(msg);                                     \
        } while (0)
#define VNASSERT(exp, vp, msg)                                          \
        do {                                                            \
            if (__predict_false(!(exp))) {                              \
                kassert_panic(msg);                                     \
            }                                                           \
        } while (0)
#else
#define KASSERT(exp,msg)        (void)(0)
#define VNASSERT(exp, vp, msg)  (void)(0)
#endif

#ifdef  ASSERT
#undef  ASSERT
#endif

#define ASSERT(exp, msg)                                                \
        do {                                                            \
            if (__predict_false(!(exp)))                                \
                kassert_panic(msg);                                     \
        } while (0)

#ifndef CTASSERT        /* Allow lint to override */
#define CTASSERT(x)     _Static_assert(x, "compile-time assertion failed")
#endif

extern int iosize_max_clamp;
#define IOSIZE_MAX      (iosize_max_clamp ? INT_MAX : SSIZE_MAX)

/* per page storage */
void    vtab_set(void* addr, uintptr_t value);
uintptr_t vtab_get(void* addr);

#define ovbcopy(f, t, l) bcopy((f), (t), (l))
void    bcopy(const void *from, void *to, size_t len) __nonnull(1) __nonnull(2);
void    bzero(void *buf, size_t len) __nonnull(1);

/* device related */
struct cdev;
const char *devtoname(struct cdev *cdev);

__END_DECLS

#endif /* _KERNEL */

#endif /* !_SYS_SYSTM_H_ */
