﻿/*-
 * Copyright (c) 2006 John Baldwin <jhb@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#ifndef _SYS_RWLOCK_H_
#define _SYS_RWLOCK_H_

#include <sys/_lock.h>
#include <sys/_rwlock.h>

#ifdef _KERNEL

/*
 * The rw_lock field consists of several fields.  The low bit indicates
 * if the lock is locked with a read (shared) or write (exclusive) lock.
 * A value of 0 indicates a write lock, and a value of 1 indicates a read
 * lock.  Bit 1 is a boolean indicating if there are any threads waiting
 * for a read lock.  Bit 2 is a boolean indicating if there are any threads
 * waiting for a write lock.  The rest of the variable's definition is
 * dependent on the value of the first bit.  For a write lock, it is a
 * pointer to the thread holding the lock, similar to the mtx_lock field of
 * mutexes.  For read locks, it is a count of read locks that are held.
 *
 * When the lock is not locked by any thread, it is encoded as a read lock
 * with zero waiters.
 */

#define	RW_LOCK_READ		0x01
#define	RW_LOCK_READ_WAITERS	0x02
#define	RW_LOCK_WRITE_WAITERS	0x04
#define	RW_LOCK_WRITE_SPINNER	0x08
#define	RW_LOCK_FLAGMASK						                    \
    (RW_LOCK_READ | RW_LOCK_READ_WAITERS | RW_LOCK_WRITE_WAITERS |	\
    RW_LOCK_WRITE_SPINNER)
#define	RW_LOCK_WAITERS		(RW_LOCK_READ_WAITERS | RW_LOCK_WRITE_WAITERS)

#define	RW_OWNER(x)		    ((x) & ~RW_LOCK_FLAGMASK)
#define	RW_READERS_SHIFT	4
#define	RW_READERS(x)		(RW_OWNER((x)) >> RW_READERS_SHIFT)
#define	RW_READERS_LOCK(x)	((x) << RW_READERS_SHIFT | RW_LOCK_READ)
#define	RW_ONE_READER		(1 << RW_READERS_SHIFT)

#define	RW_UNLOCKED		    RW_READERS_LOCK(0)
#define	RW_DESTROYED		(RW_LOCK_READ_WAITERS | RW_LOCK_WRITE_WAITERS)

void rw_init_l(struct rwlock *rw, const char *name, int opts);
void rw_init_flags_l(struct rwlock *rw, const char *name, int opts);
void rw_destroy_l(struct rwlock *rw);
void rw_rlock_l(struct rwlock *rw);
void rw_wlock_l(struct rwlock *rw);
void rw_runlock_l(struct rwlock *rw);
void rw_wunlock_l(struct rwlock *rw);
void rw_unlock_l(struct rwlock *rw);
int  rw_try_wlock_l(struct rwlock *rw);
void rw_downgrade_l(struct rwlock *rw);
void rw_sysinit(void *arg);
int  rw_initialized_l(const struct rwlock *rw);

#define rw_init(rw, name)               rw_init_l((struct rwlock*)(rw), name, 0)
#define rw_init_flags(rw, name, opts)   rw_init_flags_l((struct rwlock *)(rw), name, opts)
#define rw_destroy(rw)                  rw_destroy_l((struct rwlock *)(rw))
#define rw_rlock(rw)                    rw_rlock_l((struct rwlock *)(rw))
#define rw_wlock(rw)                    rw_wlock_l((struct rwlock *)(rw))
#define rw_runlock(rw)                  rw_runlock_l((struct rwlock *)(rw))
#define rw_wunlock(rw)                  rw_wunlock_l((struct rwlock *)(rw))
#define rw_unlock(rw)                   rw_unlock_l((struct rwlock *)(rw))
#define rw_try_wlock(rw)                rw_try_wlock_l((struct rwlock *)(rw))
#define rw_downgrade(rw)                rw_downgrade_l((struct rwlock *)(rw))
#define rw_initialized(rw)              rw_initialized_l((const struct rwlock *)(rw))

struct rw_args {
    void		*ra_rw;
    const char 	*ra_desc;
};

struct rw_args_flags {
    void		*ra_rw;
    const char 	*ra_desc;
    int		    ra_flags;
};

#define	RW_SYSINIT(name, rw, desc)					                \
    static struct rw_args name##_args = {				            \
        (rw),							                            \
        (desc),							                            \
    };								                                \
    SYSINIT(name##_rw_sysinit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	    \
        rw_sysinit, &name##_args);					                \
    SYSUNINIT(name##_rw_sysuninit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	\
        rw_destroy_l, __DEVOLATILE(void *, &(rw)->rw_lock))


#define	RW_SYSINIT_FLAGS(name, rw, desc, flags)				        \
    static struct rw_args_flags name##_args = {			            \
        (rw),							                            \
        (desc),							                            \
        (flags),						                            \
    };								                                \
    SYSINIT(name##_rw_sysinit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	    \
        rw_sysinit_flags, &name##_args);				            \
    SYSUNINIT(name##_rw_sysuninit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	\
        rw_destroy_l, __DEVOLATILE(void *, &(rw)->rw_lock))

/*
 * Options passed to rw_init_flags().
 */
#define	RW_DUPOK	        0x01
#define	RW_NOPROFILE	    0x02
#define	RW_NOWITNESS	    0x04
#define	RW_QUIET	        0x08
#define	RW_RECURSE	        0x10

#define	RA_LOCKED		    LA_LOCKED
#define	RA_RLOCKED		    LA_SLOCKED
#define	RA_WLOCKED		    LA_XLOCKED
#define	RA_UNLOCKED		    LA_UNLOCKED
#define	RA_RECURSED		    LA_RECURSED
#define	RA_NOTRECURSED		LA_NOTRECURSED

#if defined(INVARIANTS) || defined(INVARIANT_SUPPORT)
void _rw_assert(const struct rwlock *c, int what, const char *file, int line);
#define rw_assert(rw, what) _rw_assert(rw, what, __FILE__, __LINE__)
#else
#define rw_assert(rw, what) (void)(0)
#endif

#endif /* _KERNEL */
#endif /* !_SYS_RWLOCK_H_ */
