﻿
#ifndef _SYS_MUTEX_H_
#define _SYS_MUTEX_H_

#include <sys/_lock.h>
#include <sys/_mutex.h>

#ifdef _KERNEL

/*
 * Mutex types and options passed to mtx_init().  MTX_QUIET and MTX_DUPOK
 * can also be passed in.
 */
#define	MTX_DEF		    0x00000000	/* DEFAULT (sleep) lock */
#define MTX_SPIN        0x00000001      /* Spin lock (disables interrupts) */
#define MTX_RECURSE     0x00000004      /* Option: lock allowed to recurse */
#define MTX_NOWITNESS   0x00000008      /* Don't do any witness checking. */
#define MTX_NOPROFILE   0x00000020      /* Don't profile this lock */

/*
 * Option flags passed to certain lock/unlock routines, through the use
 * of corresponding mtx_{lock,unlock}_flags() interface macros.
 */
#define MTX_QUIET       LOP_QUIET       /* Don't log a mutex event */
#define MTX_DUPOK       LOP_DUPOK       /* Don't log a duplicate acquire */

/*
 * State bits kept in mutex->mtx_lock, for the DEFAULT lock type. None of this,
 * with the exception of MTX_UNOWNED, applies to spin locks.
 */
#define MTX_RECURSED    0x00000001      /* lock recursed (for MTX_DEF only) */
#define MTX_CONTESTED   0x00000002      /* lock contested (for MTX_DEF only) */
#define MTX_UNOWNED     0x00000004      /* Cookie for free mutex */
#define MTX_FLAGMASK    (MTX_RECURSED | MTX_CONTESTED | MTX_UNOWNED)

/*
 * Mutex assert flags
 */
#define MA_OWNED        LA_XLOCKED
#define MA_NOTOWNED     LA_UNLOCKED
#define MA_RECURSED     LA_RECURSED
#define MA_NOTRECURSED  LA_NOTRECURSED
#define mtx_recurse     lock_object.lo_data

#define mtx_destroy(m)          mtx_destroy_l((struct mtx*)(m))
#define mtx_init(m, name, type, opts) mtx_init_l((struct mtx*)(m), name, type, opts)
#define mtx_lock(m)             mtx_lock_l((struct mtx*)(m))
#define mtx_unlock(m)           mtx_unlock_l((struct mtx*)(m))
#define mtx_assert(m, what)     mtx_assert_l((struct mtx*)(m), what)
#define mtx_trylock(m)          mtx_trylock_l((struct mtx*)(m))
#define mtx_owned(m)            mtx_owned_l((struct mtx*)(m))
#define mtx_lock_flags(m, opt)  mtx_lock_flags_l((struct mtx*)(m), opt)

struct mtx_pool;
extern struct mtx Giant;

struct mtx_pool *mtx_pool_create(const char *mtx_name, int pool_size, int opts);
void   mtx_pool_destroy(struct mtx_pool **poolp);
struct mtx *mtx_pool_find(struct mtx_pool *pool, void *ptr);
struct mtx *mtx_pool_alloc(struct mtx_pool *pool);

#define mtx_pool_lock(pool, ptr)                                        \
        mtx_lock(mtx_pool_find((pool), (ptr)))
#define mtx_pool_lock_spin(pool, ptr)                                   \
        mtx_lock_spin(mtx_pool_find((pool), (ptr)))
#define mtx_pool_unlock(pool, ptr)                                      \
        mtx_unlock(mtx_pool_find((pool), (ptr)))
#define mtx_pool_unlock_spin(pool, ptr)                                 \
        mtx_unlock_spin(mtx_pool_find((pool), (ptr)))

void	mtx_destroy_l(struct mtx *m);
void	mtx_sysinit(void *arg);
void    mtx_init_l(struct mtx *m, const char *name, const char *type, int opts);
int     mtx_trylock_l(struct mtx *m);
void    mtx_lock_l(struct mtx *m);
void    mtx_unlock_l(struct mtx *m);
void    mtx_assert_l(struct mtx *m, int what);
int     mtx_owned_l(struct mtx *m);
void    mtx_lock_flags_l(struct mtx *m, int opt);

#define mtx_sleep(chan, mtx, pri, wmesg, timo)                      \
        _sleep((chan), &(mtx)->lock_object, (pri), (wmesg),         \
        tick_sbt * (timo), 0, C_HARDCLOCK)

struct mtx_args {
    void		*ma_mtx;
    const char 	*ma_desc;
    int		    ma_opts;
};


#define	MTX_SYSINIT(name, mtx, desc, opts)                          \
    static struct mtx_args name##_args = {				            \
        (mtx),							                            \
        (desc),							                            \
        (opts)							                            \
    };								                                \
    SYSINIT(name##_mtx_sysinit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	    \
        mtx_sysinit, &name##_args);					                \
    SYSUNINIT(name##_mtx_sysuninit, SI_SUB_LOCK, SI_ORDER_MIDDLE,	\
        mtx_destroy_l, __DEVOLATILE(void *, &(mtx)->mtx_lock))

#endif	/* _KERNEL */
#endif	/* _SYS_MUTEX_H_ */
