﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#pragma once

 /* local process id */
#define DEFAULT_PID 0

enum ClientFlags
{
    FLAG_VALID = 0x01   /* whether it is valid to do any operations on the client */
};

#ifdef _KERNEL

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/socket/socket_Statistics.h>
#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/ucred.h>
#include <sys/jail.h>
#include <sys/proc.h>
#include <sys/resourcevar.h>
#include <sys/smp.h>
#include <sys/filedesc.h>
#include <sys/file.h>
#include <sys/queue.h>

__BEGIN_DECLS

#define CLIENT_TERMINATE_POLL_INTERVAL 25000

/* list of clients */
SLIST_HEAD(slisthead, client);

struct client_config {
    uint32_t version;
    int32_t  gid;
    uint32_t tcpInitialSendBufferSize;
    uint32_t tcpInitialReceiveBufferSize;
    uint32_t tcpAutoSendBufferSizeMax;
    uint32_t tcpAutoReceiveBufferSizeMax;
    uint32_t udpSendBufferSize;
    uint32_t udpReceiveBufferSize;
    uint32_t socketBufferEfficiency;
};

struct client {
    SLIST_ENTRY(client) cl_entry;

    /* siglo related data */
    uint64_t            cl_pid;                                 /* siglo pid for this client */
    void*               cl_private_data;                        /* client's data, not used in the stack */
    void*               cl_mempool;                             /* memory pool for client allocations */
    size_t              cl_mempool_size;                        /* memory pool size for client allocations */
    nn::mem::StandardAllocator cl_allocator;                    /* siglo allocator backing up malloc */
    uintptr_t*          cl_vtab_entries;                        /* per-client v-table */
    uint32_t            cl_flags;                               /* special flags */

    /* bsd related data */
    gid_t               cl_gids[1];                             /* client gid */
    #define fdmap_len   (maxfilesperproc / CHAR_BIT / sizeof(NDSLOTTYPE))
    NDSLOTTYPE          cl_fdmap[fdmap_len];                    /* bitmap for allocated fd's */
    struct prison       cl_prison;                              /* jail, not used */
    struct uidinfo      cl_uidinfo;                             /* client uid */
    struct filedesc     cl_filedesc;                            /* top level file descriptor*/
    struct filedescent  cl_filedescent[maxfilesperproc];        /* per file descriptor entry */
    struct plimit       cl_plimit;                              /* process limits */
    struct ucred        cl_ucred;                               /* process credentials */
    struct proc         cl_proc;                                /* bsd process structure */
    struct malloc_type  cl_malloc;                              /* bsd malloc structure */
    char                cl_name[32];                            /* client name */
};

struct client* client_acquire_ref(uint64_t pid);
void           client_release_ref();
void           client_release_ref_by_pid(uint64_t pid);
int            client_create(uint64_t pid, void* mempool, size_t mempool_size, void* private_data, const client_config* p_config);
int            client_terminate(uint64_t pid, void** private_data);
void           client_free(void* container);
struct client* client_find_by_mempool(void* addr);
int            client_check_is_valid_from_cred(struct ucred * cred);
int            client_shutdown_all_sockets(int* pcount, int forced);
int            client_get_resource_statistics(uint64_t pid, nn::socket::StatisticsType type, void* buffer, size_t bufferLength, uint32_t options);

/* startup process client */
extern struct client* client0;

__END_DECLS

static inline int clientCreate(uint64_t pid, void* mempool, size_t mempool_size, void* private_data, const client_config* p_config)
{
    return client_create(pid, mempool, mempool_size, private_data, p_config);
}

static inline int clientTerminate(uint64_t pid, void** private_data)
{
    return client_terminate(pid, private_data);
}
#ifdef __SigloBSD__
static inline int clientShutdownAllSockets(int* count, int forced)
{
    return client_shutdown_all_sockets(count, forced);
}
#endif

#endif
