﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include <nn/nn_BitTypes.h>
#include <arm_neon.h>
#include "ntd-test-tls.h"

extern "C" void CallNro(void);

#define NNMUSL_SUPPORT_FLOAT16 0
#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

static const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

static nn::os::ThreadType      g_Thread1;
static nn::os::ThreadType      g_Thread2;

thread_local bool       GameBool = true;
thread_local char       GameChar = 1;
thread_local short      GameShort = 2;
thread_local int        GameInt = 100;
thread_local signed int GameSignedInt = -100;
thread_local unsigned int GameUnsignedInt = 102;
thread_local __fp16     Game__fp16 = -4.0;
#if NNMUSL_SUPPORT_FLOAT16
thread_local _Float16   Game_Float16 = -5.0;
#endif
thread_local float      GameFloat = -3.0;
thread_local double     GameDouble = 2.0;
thread_local char       *GameCharPtrToStringLit = (char *)"Five";
thread_local char       GameCharArray[20] = {'s', 'i', 'x', 0};
thread_local char       *GameCharPtrToCharArray = (char *)GameCharArray;
typedef union TLS_Union {
    char age;
    double weight;
} TLS_Union;
thread_local TLS_Union GameUnion1 = {101};
thread_local TLS_Union GameUnion2 = {.weight = 123.5};
typedef struct TLS_Struct {
    char age;
    double weight;
} TLS_Struct;
thread_local TLS_Struct GameStruct1 = {101};
thread_local TLS_Struct GameStruct2 = {101, 124.5};
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
thread_local nn::Bit128 GameBit128 = (__int128)0xDEADF00D;
#else
thread_local nn::Bit128 GameBit128 = {0,(uint64_t)0xDEADF00D};
#endif
typedef int v2si __attribute__((vector_size(8)));
thread_local v2si       GameVector1 = {1,1};
thread_local int32x2_t  GameVector2 = {2,2};

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#define NRO_ROOT_NAME STRINGIZE_VALUE_OF(TLS_NRO_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"
#define NR0_NAME "rom:/nro/" NRO_ROOT_NAME ".nro"

static size_t ReadAll(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

static const size_t MaxFileSize  = 0x400000;
static void* nro;
static void* bss;
static void* nrr;
static size_t imageSize;
static size_t bufferSize;
static size_t nrrSize;
static nn::ro::Module module;
static nn::ro::RegistrationInfo info;
static long            num_dtrs_to_call = 0;
static char* cacheBuffer;

static void load_nro_module()
{
    nn::Result result;

    TESTLOG("load_nro_module calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    size_t cacheSize = 0;
    TESTLOG("load_nro_module calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("load_nro_module cacheSize is %zd", cacheSize);

    // キャッシュバッファを確保
    TESTLOG("load_nro_module calling new(std::nothrow) char[cacheSize]");
    cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("load_nro_module calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // nrr を読み込み
    TESTLOG("load_nro_module calling nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module nrr is %p and was allocated to be MaxFileSize = %zd", nrr, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nrr, MaxFileSize, \"%s\")", NRR_NAME);
    nrrSize = ReadAll(nrr, MaxFileSize, NRR_NAME);
    TESTLOG("load_nro_module nrrSize is %zd", nrrSize);

    // nrr を登録
    TESTLOG("load_nro_module calling nn::ro::RegisterModuleInfo(&info, nrr)");
    result = nn::ro::RegisterModuleInfo(&info, nrr);
    NN_ASSERT(result.IsSuccess());

    TESTLOG("load_nro_module calling nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module nro is %p and was allocated to be MaxFileSize = %zd", nro, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nro, MaxFileSize, \"%s\")", NR0_NAME);
    imageSize = ReadAll(nro, MaxFileSize, NR0_NAME);
    TESTLOG("load_nro_module imageSize is %zd", imageSize);

    TESTLOG("load_nro_module calling nn::ro::GetBufferSize(&bufferSize, nro)");
    result = nn::ro::GetBufferSize(&bufferSize, nro);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("load_nro_module bufferSize is %zd", bufferSize);
    if (bufferSize != 0)
    {
    TESTLOG("load_nro_module calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
        bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
    }
    else
    {
    TESTLOG("load_nro_module setting bss = 0");
        bss = 0;
    }

    // nro をロード(シンボルは遅延解決)
    TESTLOG("load_nro_module calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
    result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
    NN_ASSERT(result.IsSuccess());
}

static void unload_nro_module()
{
    TESTLOG("unload_nro_module calling nn::ro::UnloadModule(&module)");
    nn::ro::UnloadModule(&module);
    TESTLOG("unload_nro_module calling nn::ro::UnregisterModuleInfo(&info)");
    nn::ro::UnregisterModuleInfo(&info);
    TESTLOG("unload_nro_module calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("unload_nro_module calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;
}

static void static_func()
{
    int GamePlusCharPtrToStringLit_strcmp, GamePlusCharPtrToCharArray_strcmp, Bit128_memcmp;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit128 test1 = (__int128)0xDEADBEEF;
#else
    nn::Bit128 test1 = {0,(uint64_t)0xDEADBEEF};
#endif
    int *array;

    thread_local bool       GameBool = false;
    thread_local char       GameChar = 10;
    thread_local short      GameShort = 20;
    thread_local int        GameInt = 1000;
    thread_local signed int GameSignedInt = -1000;
    thread_local unsigned int GameUnsignedInt = 1002;
    thread_local __fp16     Game__fp16 = -6.0;
#if NNMUSL_SUPPORT_FLOAT16
    thread_local _Float16   Game_Float16 = -7.0;
#endif
    thread_local float      GameFloat = -30.0;
    thread_local double     GameDouble = 20.0;
    thread_local char       *GameCharPtrToStringLit = (char *)"Seven";
    thread_local char       GameCharArray[20] = {'e', 'i', 'g', 'h', 't', 0};
    thread_local char       *GameCharPtrToCharArray = (char *)GameCharArray;
    thread_local TLS_Union GameUnion1 = {13};
    thread_local TLS_Union GameUnion2 = {.weight = 12.5};
    thread_local TLS_Struct GameStruct1 = {13};
    thread_local TLS_Struct GameStruct2 = {12, 14.5};
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    thread_local nn::Bit128 GameBit128 = (__int128)0xDEADBEEF;
#else
    thread_local nn::Bit128 GameBit128 = {0,(uint64_t)0xDEADBEEF};
#endif
    thread_local v2si       GameVector1 = {10,11};
    thread_local int32x2_t  GameVector2 = {12,9};

    TESTCASE_MESSAGE(GameChar == 10, "GameChar when initialized: %d (expected 10)", GameChar);
    TESTCASE_MESSAGE(GameShort == 20, "GameShort when initialized: %d (expected 20)", GameShort);
    TESTCASE_MESSAGE(GameInt == 1000, "GameInt when initialized: %d (expected 1000)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 20.0, "GameDouble when initialized: %d (expected 20.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -30.0, "GameFloat when initialized: %f (expected -30.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -6.0, "Game__fp16 when initialized: %f (expected -6.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -7.0, "Game_Float16 when initialized: %f (expected -7.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "Seven");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"Seven\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"Seven\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "eight", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"eight\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"eight\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == false, "GameBool when initialized: %d (expected false)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -1000, "GameSignedInt when initialized: %d (expected -1000)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 1002, "GameUnsignedInt when initialized: %d (expected 1002)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.age == 13, "GameUnion1.age when initialized: %d (expected 13)", GameUnion1.age);
    TESTCASE_MESSAGE(GameUnion2.weight == 12.5, "GameUnion2.weight when initialized: %f (expected 12.5)", GameUnion2.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 13, "GameStruct1.age when initialized: %d (expected 13)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 0.0, "GameStruct1.weight when initialized: %f (expected 0.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 12, "GameStruct2.age when initialized: %d (expected 12)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 14.5, "GameStruct2.weight when initialized: %f (expected 14.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test1, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 10) && (array[1] == 11), "expect {%d,%d} but got {%d,%d}", 10, 11, array[0], array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == 12) && (array[1] == 9), "expect {%d,%d} but got {%d,%d}", 12, 9, array[0], array[1]);
}

static inline void static_inline_func()
{
    int GamePlusCharPtrToStringLit_strcmp, GamePlusCharPtrToCharArray_strcmp, Bit128_memcmp;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit128 test1 = (__int128)0xCAFEBEEF;
#else
    nn::Bit128 test1 = {0,(uint64_t)0xCAFEBEEF};
#endif
    int *array;

    thread_local bool       GameBool = true;
    thread_local char       GameChar = 100;
    thread_local short      GameShort = 200;
    thread_local int        GameInt = 10000;
    thread_local signed int GameSignedInt = -10000;
    thread_local unsigned int GameUnsignedInt = 10020;
    thread_local __fp16     Game__fp16 = -8.0;
#if NNMUSL_SUPPORT_FLOAT16
    thread_local _Float16   Game_Float16 = -9.0;
#endif
    thread_local float      GameFloat = -300.0;
    thread_local double     GameDouble = 200.0;
    thread_local char       *GameCharPtrToStringLit = (char *)"Nine";
    thread_local char       GameCharArray[20] = {'t', 'e', 'n', 0};
    thread_local char       *GameCharPtrToCharArray = (char *)GameCharArray;
    thread_local TLS_Union GameUnion1 = {14};
    thread_local TLS_Union GameUnion2 = {.weight = 13.5};
    thread_local TLS_Struct GameStruct1 = {14};
    thread_local TLS_Struct GameStruct2 = {13, 15.5};
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    thread_local nn::Bit128 GameBit128 = (__int128)0xCAFEBEEF;
#else
    thread_local nn::Bit128 GameBit128 = {0,(uint64_t)0xCAFEBEEF};
#endif
    thread_local v2si       GameVector1 = {11,12};
    thread_local int32x2_t  GameVector2 = {3,-1};

    TESTCASE_MESSAGE(GameChar == 100, "GameChar when initialized: %d (expected 100)", GameChar);
    TESTCASE_MESSAGE(GameShort == 200, "GameShort when initialized: %d (expected 200)", GameShort);
    TESTCASE_MESSAGE(GameInt == 10000, "GameInt when initialized: %d (expected 10000)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 200.0, "GameDouble when initialized: %d (expected 200.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -300.0, "GameFloat when initialized: %f (expected -300.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -8.0, "Game__fp16 when initialized: %f (expected -8.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -9.0, "Game_Float16 when initialized: %f (expected -9.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "Nine");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"Nine\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"Nine\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "ten", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"ten\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"ten\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == true, "GameBool when initialized: %d (expected true)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -10000, "GameSignedInt when initialized: %d (expected -10000)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 10020, "GameUnsignedInt when initialized: %d (expected 10020)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.age == 14, "GameUnion1.age when initialized: %d (expected 14)", GameUnion1.age);
    TESTCASE_MESSAGE(GameUnion2.weight == 13.5, "GameUnion2.weight when initialized: %f (expected 13.5)", GameUnion2.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 14, "GameStruct1.age when initialized: %d (expected 14)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 0.0, "GameStruct1.weight when initialized: %f (expected 0.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 13, "GameStruct2.age when initialized: %d (expected 13)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 15.5, "GameStruct2.weight when initialized: %f (expected 15.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test1, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 11) && (array[1] == 12), "expect {%d,%d} but got {%d,%d}", 11, 12, array[0], array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == 3) && (array[1] == -1), "expect {%d,%d} but got {%d,%d}", 3, -1, array[0], array[1]);
}

namespace {
inline void namespace_inline_func()
{
    int GamePlusCharPtrToStringLit_strcmp, GamePlusCharPtrToCharArray_strcmp, Bit128_memcmp;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit128 test1 = (__int128)0xCAFECAFE;
#else
    nn::Bit128 test1 = {0,(uint64_t)0xCAFECAFE};
#endif
    int *array;

    thread_local bool       GameBool = true;
    thread_local char       GameChar = 50;
    thread_local short      GameShort = 201;
    thread_local int        GameInt = 10001;
    thread_local signed int GameSignedInt = -10001;
    thread_local unsigned int GameUnsignedInt = 10021;
    thread_local __fp16     Game__fp16 = -81.0;
#if NNMUSL_SUPPORT_FLOAT16
    thread_local _Float16   Game_Float16 = -91.0;
#endif
    thread_local float      GameFloat = -301.0;
    thread_local double     GameDouble = 201.0;
    thread_local char       *GameCharPtrToStringLit = (char *)"Eleven";
    thread_local char       GameCharArray[20] = {'t', 'w', 'e', 'l', 'v', 'e', 0};
    thread_local char       *GameCharPtrToCharArray = (char *)GameCharArray;
    thread_local TLS_Union GameUnion1 = {75};
    thread_local TLS_Union GameUnion2 = {.weight = 131.5};
    thread_local TLS_Struct GameStruct1 = {16};
    thread_local TLS_Struct GameStruct2 = {5, 0.5};
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    thread_local nn::Bit128 GameBit128 = (__int128)0xCAFECAFE;
#else
    thread_local nn::Bit128 GameBit128 = {0,(uint64_t)0xCAFECAFE};
#endif
    thread_local v2si       GameVector1 = {81,1};
    thread_local int32x2_t  GameVector2 = {5,-111};

    TESTCASE_MESSAGE(GameChar == 50, "GameChar when initialized: %d (expected 50)", GameChar);
    TESTCASE_MESSAGE(GameShort == 201, "GameShort when initialized: %d (expected 201)", GameShort);
    TESTCASE_MESSAGE(GameInt == 10001, "GameInt when initialized: %d (expected 10001)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 201.0, "GameDouble when initialized: %d (expected 200.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -301.0, "GameFloat when initialized: %f (expected -300.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -81.0, "Game__fp16 when initialized: %f (expected -81.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -91.0, "Game_Float16 when initialized: %f (expected -91.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "Eleven");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"Eleven\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"Eleven\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "twelve", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"twelve\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"twelve\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == true, "GameBool when initialized: %d (expected true)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -10001, "GameSignedInt when initialized: %d (expected -10001)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 10021, "GameUnsignedInt when initialized: %d (expected 10021)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.age == 75, "GameUnion1.age when initialized: %d (expected 75)", GameUnion1.age);
    TESTCASE_MESSAGE(GameUnion2.weight == 131.5, "GameUnion2.weight when initialized: %f (expected 131.5)", GameUnion2.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 16, "GameStruct1.age when initialized: %d (expected 16)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 0.0, "GameStruct1.weight when initialized: %f (expected 0.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 5, "GameStruct2.age when initialized: %d (expected 5)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 0.5, "GameStruct2.weight when initialized: %f (expected 0.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test1, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 81) && (array[1] == 1), "expect {%d,%d} but got {%d,%d}", 81, 1, array[0], array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == 5) && (array[1] == -111), "expect {%d,%d} but got {%d,%d}", 5, -111, array[0], array[1]);
}
}

/*---------------------------------------------------------------------------*
  Name:         TestThreadOne

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThreadOne(void *arg)
{
    int GamePlusCharPtrToStringLit_strcmp, GamePlusCharPtrToCharArray_strcmp, Bit128_memcmp;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit128 test1 = (__int128)0xDEADF00D;
    nn::Bit128 test2 = (__int128)0xDEADF00E;
#else
    nn::Bit128 test1 = {0,(uint64_t)0xDEADF00D};
    nn::Bit128 test2 = {0,(uint64_t)0xDEADF00E};
#endif
    int *array;

    TESTLOG("TestThreadOne arg is %s", (char *)arg);

    TESTCASE_MESSAGE(GameChar == 1, "GameChar when initialized: %d (expected 1)", GameChar);
    TESTCASE_MESSAGE(GameShort == 2, "GameShort when initialized: %d (expected 2)", GameShort);
    TESTCASE_MESSAGE(GameInt == 100, "GameInt when initialized: %d (expected 100)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 2.0, "GameDouble when initialized: %d (expected 2.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -3.0, "GameFloat when initialized: %f (expected -3.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -4.0, "Game__fp16 when initialized: %f (expected -4.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -5.0, "Game_Float16 when initialized: %f (expected -5.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "Five");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"Five\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"Five\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "six", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"six\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"six\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == true, "GameBool when initialized: %d (expected true)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -100, "GameSignedInt when initialized: %d (expected -100)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 102, "GameUnsignedInt when initialized: %d (expected 102)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.age == 101, "GameUnion1.age when initialized: %d (expected 101)", GameUnion1.age);
    TESTCASE_MESSAGE(GameUnion2.weight == 123.5, "GameUnion2.weight when initialized: %f (expected 123.5)", GameUnion2.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 101, "GameStruct1.age when initialized: %d (expected 101)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 0.0, "GameStruct1.weight when initialized: %f (expected 0.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 101, "GameStruct2.age when initialized: %d (expected 101)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 124.5, "GameStruct2.weight when initialized: %f (expected 124.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test1, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 1) && (array[1] == 1), "expect {%d,%d} but got {%d,%d}", 1,1,array[0],array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == 2) && (array[1] == 2), "expect {%d,%d} but got {%d,%d}", 2, 2, array[0], array[1]);

    GameChar = 6;
    GameShort = 30;
    GameInt = 605;
    GameFloat = -25.0;
    GameDouble = 27.0;
    Game__fp16++;
#if NNMUSL_SUPPORT_FLOAT16
    Game_Float16--;
#endif

    GameCharPtrToStringLit++;
    GameCharPtrToCharArray++;

    GameBool = false;
    GameSignedInt--;
    GameUnsignedInt++;
    GameUnion1.weight = 29.0;
    GameStruct1.age = 99;
    GameStruct1.weight = 10.0;
    GameStruct2.age--;
    GameStruct2.weight++;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    GameBit128++;
#else
    GameBit128 += {0,1};
#endif
    GameVector1 += {2,3};
    GameVector2 *= -1;

    static_func();
    static_inline_func();
    namespace_inline_func();

    TESTCASE_MESSAGE(GameShort == 30, "GameShort at thread exit: %d (expected 30)", GameShort);
    TESTCASE_MESSAGE(GameChar == 6, "GameChar at thread exit: %d (expected 6)", GameChar);
    TESTCASE_MESSAGE(GameInt == 605, "GameInt at thread exit: %d (expected 605)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 27.0, "GameDouble at thread exit: %f (expected 27.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -25.0, "GameFloat at thread exit: %f (expected -25.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -3.0, "Game__fp16 when initialized: %f (expected -3.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -6.0, "Game_Float16 when initialized: %f (expected -6.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "ive");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"ive\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"ive\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "ix", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"ix\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"ix\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == false, "GameBool when initialized: %d (expected false)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -101, "GameSignedInt when initialized: %d (expected -101)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 103, "GameUnsignedInt when initialized: %d (expected 103)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.weight == 29.0, "GameUnion1.weight when initialized: %f (expected 29.0)", GameUnion1.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 99, "GameStruct1.age when initialized: %d (expected 99)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 10.0, "GameStruct1.weight when initialized: %f (expected 10.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 100, "GameStruct2.age when initialized: %d (expected 100)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 125.5, "GameStruct2.weight when initialized: %f (expected 125.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test2, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 3) && (array[1] == 4), "expect {%d,%d} but got {%d,%d}", 3,4,array[0],array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == -2) && (array[1] == -2), "expect {%d,%d} but got {%d,%d}", -2, -2, array[0], array[1]);
}


/*---------------------------------------------------------------------------*
  Name:         TestThreadTwo

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/

static void TestThreadTwo(void *arg)
{
    int GamePlusCharPtrToStringLit_strcmp, GamePlusCharPtrToCharArray_strcmp, Bit128_memcmp;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit128 test1 = (__int128)0xDEADF00D;
    nn::Bit128 test2 = (__int128)0xDEADF00E;
#else
    nn::Bit128 test1 = {0,(uint64_t)0xDEADF00D};
    nn::Bit128 test2 = {0,(uint64_t)0xDEADF00E};
#endif
    int *array;

    TESTLOG("TestThreadTwo arg is %s", (char *)arg);

    TESTCASE_MESSAGE(GameChar == 1, "GameChar when initialized: %d (expected 1)", GameChar);
    TESTCASE_MESSAGE(GameShort == 2, "GameShort when initialized: %d (expected 2)", GameShort);
    TESTCASE_MESSAGE(GameInt == 100, "GameInt when initialized: %d (expected 100)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 2.0, "GameDouble when initialized: %d (expected 2.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -3.0, "GameFloat when initialized: %f (expected -3.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -4.0, "Game__fp16 when initialized: %f (expected -4.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -5.0, "Game_Float16 when initialized: %f (expected -5.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "Five");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"Five\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"Five\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "six", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"six\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"six\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == true, "GameBool when initialized: %d (expected true)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -100, "GameSignedInt when initialized: %d (expected -100)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 102, "GameUnsignedInt when initialized: %d (expected 102)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.age == 101, "GameUnion1.age when initialized: %d (expected 101)", GameUnion1.age);
    TESTCASE_MESSAGE(GameUnion2.weight == 123.5, "GameUnion2.weight when initialized: %f (expected 123.5)", GameUnion2.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 101, "GameStruct1.age when initialized: %d (expected 101)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 0.0, "GameStruct1.weight when initialized: %f (expected 0.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 101, "GameStruct2.age when initialized: %d (expected 101)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 124.5, "GameStruct2.weight when initialized: %f (expected 124.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test1, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 1) && (array[1] == 1), "expect {%d,%d} but got {%d,%d}", 1,1,array[0],array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == 2) && (array[1] == 2), "expect {%d,%d} but got {%d,%d}", 2, 2, array[0], array[1]);

    GameChar = 6;
    GameShort = 30;
    GameInt = 605;
    GameFloat = -25.0;
    GameDouble = 27.0;
    Game__fp16++;
#if NNMUSL_SUPPORT_FLOAT16
    Game_Float16--;
#endif

    GameCharPtrToStringLit++;
    GameCharPtrToCharArray++;

    GameBool = false;
    GameSignedInt--;
    GameUnsignedInt++;
    GameUnion1.weight = 29.0;
    GameStruct1.age = 99;
    GameStruct1.weight = 10.0;
    GameStruct2.age--;
    GameStruct2.weight++;
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    GameBit128++;
#else
    GameBit128 += {0,1};
#endif
    GameVector1 += {2,3};
    GameVector2 *= -1;

    static_func();
    static_inline_func();
    namespace_inline_func();

    TESTCASE_MESSAGE(GameShort == 30, "GameShort at thread exit: %d (expected 30)", GameShort);
    TESTCASE_MESSAGE(GameChar == 6, "GameChar at thread exit: %d (expected 6)", GameChar);
    TESTCASE_MESSAGE(GameInt == 605, "GameInt at thread exit: %d (expected 605)", GameInt);
    TESTCASE_MESSAGE(GameDouble == 27.0, "GameDouble at thread exit: %f (expected 27.0)", GameDouble);
    TESTCASE_MESSAGE(GameFloat == -25.0, "GameFloat at thread exit: %f (expected -25.0)", GameFloat);
    TESTCASE_MESSAGE(Game__fp16 == -3.0, "Game__fp16 when initialized: %f (expected -3.0)", Game__fp16);
#if NNMUSL_SUPPORT_FLOAT16
    TESTCASE_MESSAGE(Game_Float16 == -6.0, "Game_Float16 when initialized: %f (expected -6.0)", Game_Float16);
#endif

    GamePlusCharPtrToStringLit_strcmp = strcmp(GameCharPtrToStringLit, "ive");
    TESTLOG("GamePlusCharPtrToStringLit_strcmp = %d; expected 0 for strcmp(GameCharPtrToStringLit, \"ive\")", GamePlusCharPtrToStringLit_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToStringLit_strcmp == 0, "expected 0 but got %d for strcmp(GameCharPtrToStringLit, \"ive\")", GamePlusCharPtrToStringLit_strcmp);
    GamePlusCharPtrToCharArray_strcmp = strncmp(GameCharPtrToCharArray, "ix", 20);
    TESTLOG("GamePlusCharPtrToCharArray_strcmp = %d; expected 0 for strncmp(GameCharPtrToCharArray, \"ix\", 20)", GamePlusCharPtrToCharArray_strcmp);
    TESTCASE_MESSAGE(GamePlusCharPtrToCharArray_strcmp == 0, "expected 0 but got %d for strncmp(GameCharPtrToCharArray, \"ix\", 20)", GamePlusCharPtrToCharArray_strcmp);

    TESTCASE_MESSAGE(GameBool == false, "GameBool when initialized: %d (expected false)", GameBool);
    TESTCASE_MESSAGE(GameSignedInt == -101, "GameSignedInt when initialized: %d (expected -101)", GameSignedInt);
    TESTCASE_MESSAGE(GameUnsignedInt == 103, "GameUnsignedInt when initialized: %d (expected 103)", GameUnsignedInt);
    TESTCASE_MESSAGE(GameUnion1.weight == 29.0, "GameUnion1.weight when initialized: %f (expected 29.0)", GameUnion1.weight);
    TESTCASE_MESSAGE(GameStruct1.age == 99, "GameStruct1.age when initialized: %d (expected 99)", GameStruct1.age);
    TESTCASE_MESSAGE(GameStruct1.weight == 10.0, "GameStruct1.weight when initialized: %f (expected 10.0)", GameStruct1.weight);
    TESTCASE_MESSAGE(GameStruct2.age == 100, "GameStruct2.age when initialized: %d (expected 100)", GameStruct2.age);
    TESTCASE_MESSAGE(GameStruct2.weight == 125.5, "GameStruct2.weight when initialized: %f (expected 125.5)", GameStruct2.weight);
    Bit128_memcmp = memcmp(&GameBit128, &test2, sizeof(nn::Bit128));
    TESTCASE_MESSAGE(Bit128_memcmp == 0, "expect 0 but got %d for memcmp(&GameBit128, &test1, sizeof(nn::Bit128))", Bit128_memcmp);
    array = (int *)&GameVector1;
    TESTCASE_MESSAGE((array[0] == 3) && (array[1] == 4), "expect {%d,%d} but got {%d,%d}", 3,4,array[0],array[1]);
    array = (int *)&GameVector2;
    TESTCASE_MESSAGE((array[0] == -2) && (array[1] == -2), "expect {%d,%d} but got {%d,%d}", -2, -2, array[0], array[1]);
}

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    load_nro_module();

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThreadOne, (void *)"1", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    result = nn::os::CreateThread( &g_Thread2, TestThreadTwo, (void *)"2", g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    TESTLOG("g_Thread2 = 0x%08zX", (size_t)&g_Thread2);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");
    nn::os::StartThread( &g_Thread2 );
    TESTLOG("g_Thread2 started");

    TestThreadOne((void *)"main_thread");
    CallNro();

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");
    nn::os::WaitThread( &g_Thread2 );
    TESTLOG("g_Thread2 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");
    nn::os::DestroyThread( &g_Thread2 );
    TESTLOG("g_Thread2 destroyed");

    unload_nro_module();

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

