﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <thread>
#include <climits>
#include <pthread.h>
#include "ntd-test-tls.h"

/* __nnmusl_call_tls_dtors <-> __cxa_thread_atexit_impl deadlock */

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static int num_TlsData1DestructorCalls = 0;
static int num_TlsData2DestructorCalls = 0;
class TlsData1
{
public:
    long TlsData1_num_dtrs_to_call;
    ~TlsData1()
    {
        TESTLOG("In ~TlsData1().");
        TlsData1_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors at the end of ~TlsData1().", TlsData1_num_dtrs_to_call);
        TESTCASE_MESSAGE(TlsData1_num_dtrs_to_call == 2, "Before exiting ~TlsData1() we expected 2 uncalled Destructors but we got %ld.",
            TlsData1_num_dtrs_to_call);
        num_TlsData1DestructorCalls++;
    }
};

class TlsData2
{
public:
    long TlsData2_num_dtrs_to_call;
    ~TlsData2()
    {
        // We get here from __nnmusl_call_tls_dtors, i.e.
        // we are in the context of a TLS destructor in T1 now.
        // Spawn the thread T2 that calls __cxa_thread_atexit_impl
        // to register ~TlsData1.
        // Call to __cxa_thread_atexit_impl in T2 within the
        // __nnmusl_call_tls_dtors context in T1 leads to the deadlock.
        TlsData2_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors at the beginning of ~TlsData2() (before spawning T2).", TlsData2_num_dtrs_to_call);
        TESTCASE_MESSAGE(TlsData2_num_dtrs_to_call == 1, "In ~TlsData2() before spawning T2, we expected 1 uncalled Destructor but we got %ld.",
            TlsData2_num_dtrs_to_call);
        TESTLOG("Spawn the thread T2 that calls __cxa_thread_atexit_impl to register ~TlsData1.");
        std::thread([]{ thread_local TlsData1 x; }).join();
        TESTLOG("Spawning T2 completed.");
        TlsData2_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors at the end of ~TlsData2() (after spawning and exiting T2).", TlsData2_num_dtrs_to_call);
        TESTCASE_MESSAGE(TlsData2_num_dtrs_to_call == 1, "Before exiting ~TlsData2()  (after spawning and exiting T2) we expected 1 uncalled Destructor but we got %ld.",
            TlsData2_num_dtrs_to_call);
        num_TlsData2DestructorCalls++;
    }
};

extern "C" void nnMain()
{
    long num_dtrs_to_call;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);
    // Spawn the worker thread T1 that enters __nnmusl_call_tls_dtors
    // to destroy "y" when it completes
    TESTLOG("Spawn the worker thread T1 that enters __nnmusl_call_tls_dtors to destroy \"y\" when it completes.");
    std::thread([]{ thread_local TlsData2 y; }).join();
    TESTLOG("Spawning T1 completed.");

    TESTLOG("num_TlsData1DestructorCalls = %d", num_TlsData1DestructorCalls);
    TESTCASE_MESSAGE(num_TlsData1DestructorCalls == 1, "Before exiting nnMain we expected 1 calls to ~TlsData1() but we got %ld.",
        num_TlsData1DestructorCalls);
    TESTLOG("num_TlsData2DestructorCalls = %d", num_TlsData2DestructorCalls);
    TESTCASE_MESSAGE(num_TlsData2DestructorCalls == 1, "Before exiting nnMain we expected 1 calls to ~TlsData2() but we got %ld.",
        num_TlsData2DestructorCalls);

    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the end of nnMain.", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before exiting nnMain we expected 0 uncalled Destructors but we got %ld.", num_dtrs_to_call);
    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}




