﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <thread>
#include <climits>
#include <pthread.h>
#include "ntd-test-tls.h"

/* __nnmusl_call_tls_dtors <-> __nnmusl_call_tls_dtors deadlock */

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static int num_TlsDataDestructorCalls = 0;
class TlsData
{
public:
    long TlsData_num_dtrs_to_call;
    ~TlsData()
    {
        // We get here from __nnmusl_call_tls_dtors, i.e.
        // we are in the context of a TLS destructor in T1 now.
        // Spawn the thread T2 that also calls
        // __nnmusl_call_tls_dtors once it completes.
        // Concurrent __nnmusl_call_tls_dtors calls from
        // T1 and T2 lead to the deadlock.
        TESTLOG("Spawn the thread T2 that also calls __nnmusl_call_tls_dtors once it completes.");
        std::thread([]{ thread_local int x; }).join();
        TESTLOG("Spawning T2 completed.");
        TlsData_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors at the end of ~TlsData().", TlsData_num_dtrs_to_call);
        TESTCASE_MESSAGE(TlsData_num_dtrs_to_call == 1, "Before exiting ~TlsData() we expected 1 uncalled Destructor but we got %ld.",
            TlsData_num_dtrs_to_call);
        num_TlsDataDestructorCalls++;
    }
};

extern "C" void nnMain()
{
    long num_dtrs_to_call;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);
    // Spawn the worker thread T1 that enters __nnmusl_call_tls_dtors
    // to destroy "y" when it completes
    TESTLOG("Spawn the worker thread T1 that enters __nnmusl_call_tls_dtors to destroy \"y\" when it completes.");
    std::thread([]{ thread_local TlsData y; }).join();
    TESTLOG("Spawning T1 completed.");

    TESTLOG("num_TlsDataDestructorCalls = %d", num_TlsDataDestructorCalls);
    TESTCASE_MESSAGE(num_TlsDataDestructorCalls == 1, "Before exiting nnMain we expected 1 calls to ~TlsData() but we got %ld.",
        num_TlsDataDestructorCalls);

    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the end of nnMain.", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before exiting nnMain we expected 0 uncalled Destructors but we got %ld.", num_dtrs_to_call);
    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}



