﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

#define MIN_ALIGN nn::os::ThreadStackAlignment

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define STACK_SIZE (3 * MIN_ALIGN)
#define NUM_THREADS 32

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ

typedef struct thread_stack {
    NN_OS_ALIGNAS_THREAD_STACK char  stack[ ThreadStackSize ];   // 1つ目のスレッドのスタック
} thread_stack;
thread_stack            g_ThreadStack[NUM_THREADS];

nn::os::ThreadType      g_Thread[NUM_THREADS];


#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static int counter = 0;
static int num_InnerHogeDestructorCalls = 0;
static int num_HogeDestructorCalls = 0;
class InnerHoge
{
public:
    int x;
    int id;
    nn::os::ThreadType *InnerHoge_thread;
    char *name = (char *)"<unitialized name>";
    InnerHoge() {
        id = ++counter;
        InnerHoge_thread = nn::os::GetCurrentThread();
        TESTLOG("Hello from class InnerHoge (%s) %d thread address 0x%08zX; id = %d", name, x, (size_t)InnerHoge_thread, id);
    }
    InnerHoge(char *func_name, int T) {
        x = T;
        id = ++counter;
        name = func_name;
        InnerHoge_thread = nn::os::GetCurrentThread();
        TESTLOG("Hello from class InnerHoge (%s) %d thread address 0x%08zX; id = %d", name, x, (size_t)InnerHoge_thread, id);
    }
    ~InnerHoge() {
        TESTLOG("Good-bye from class InnerHoge (%s) %d thread address 0x%08zX; id = %d!", name, x, (size_t)InnerHoge_thread, id);
        num_InnerHogeDestructorCalls++;
    }
};

static void InnerTestThread(void *arg)
{
    long num_dtrs_to_call_start, num_dtrs_to_call_end;
    num_dtrs_to_call_start = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the beginning of InnerTestThread %p.", num_dtrs_to_call_start, arg);

    thread_local InnerHoge g_InnerHoge00((char *)"g_InnerHoge00", 0);

    num_dtrs_to_call_end = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the end of InnerTestThread %p.", num_dtrs_to_call_end, arg);
}

class Hoge
{
public:
    int x;
    int id;
    nn::Result result;
    nn::os::ThreadType *Hoge_thread;
    char *name = (char *)"<unitialized name>";
    NN_OS_ALIGNAS_THREAD_STACK char local_stack[ ThreadStackSize ];
    nn::os::ThreadType local_Thread;
    Hoge() {
        id = ++counter;
        Hoge_thread = nn::os::GetCurrentThread();
        TESTLOG("Hello from class Hoge (%s) %d thread address 0x%08zX; id = %d", name, x, (size_t)Hoge_thread, id);

        result = nn::os::CreateThread( &local_Thread, InnerTestThread, (void *)((size_t)id), local_stack, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create local_Thread.");
        TESTLOG("local_Thread = 0x%08zX created", (size_t)&local_Thread);

        nn::os::StartThread( &local_Thread );
        TESTLOG("local_Thread = 0x%08zX started", (size_t)&local_Thread);

        nn::os::WaitThread( &local_Thread );
        TESTLOG("local_Thread = 0x%08zX did wait", (size_t)&local_Thread);
    }
    Hoge(char *func_name, int T) {
        x = T;
        id = ++counter;
        name = func_name;
        Hoge_thread = nn::os::GetCurrentThread();
        TESTLOG("Hello from class Hoge (%s) %d thread address 0x%08zX; id = %d", name, x, (size_t)Hoge_thread, id);

        result = nn::os::CreateThread( &local_Thread, InnerTestThread, (void *)((size_t)id), local_stack, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create local_Thread.");
        TESTLOG("local_Thread = 0x%08zX created", (size_t)&local_Thread);

        nn::os::StartThread( &local_Thread );
        TESTLOG("local_Thread = 0x%08zX started", (size_t)&local_Thread);

        nn::os::WaitThread( &local_Thread );
        TESTLOG("local_Thread = 0x%08zX did wait", (size_t)&local_Thread);
    }
    ~Hoge() {
        TESTLOG("Good-bye from class Hoge (%s) %d thread address 0x%08zX; id = %d!", name, x, (size_t)Hoge_thread, id);

        nn::os::DestroyThread( &local_Thread );
        TESTLOG("local_Thread = 0x%08zX destroyed", (size_t)&local_Thread);
        num_HogeDestructorCalls++;
    }
};


/*---------------------------------------------------------------------------*
  Name:         TestThread

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    1 or 2.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/

static void TestThread(void *arg)
{
    long num_dtrs_to_call_start, num_dtrs_to_call_end;
    num_dtrs_to_call_start = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the beginning of TestThread %p.", num_dtrs_to_call_start, arg);

    thread_local Hoge g_Hoge00((char *)"g_Hoge00", 0);

    num_dtrs_to_call_end = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the end of TestThread %p.", num_dtrs_to_call_end, arg);
}

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    size_t i;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    TESTLOG("This TLS Constructor test stress tests that a constructor will be called in the first function that has any thread_local variable referenced in it.");
    TESTLOG("The stress test defines 100 threads and each one will have 100 constructors and destructors.");
    TESTLOG("The thread_local variable has to be visible in the compilation unit but it doesn't have to be the class object.  Any variable will do.");

    nn::Result      result;
    long            num_dtrs_to_call = 0;
    nn::os::ThreadType *main_thread = nn::os::GetCurrentThread();
    TESTLOG("1: main_thread = 0x%08zX", (size_t)main_thread);
    TESTLOG("begin creating threads");
    for (i = 0; i < NUM_THREADS; i++) {
        result = nn::os::CreateThread( &(g_Thread[i]), TestThread, (void *)i, g_ThreadStack[i].stack, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread[%zd].", i );
        TESTLOG("g_Thread[%zd] = 0x%08zX created", i, (size_t)&(g_Thread[i]));
    }

    main_thread = nn::os::GetCurrentThread();
    TESTLOG("2: main_thread = 0x%08zX", (size_t)main_thread);

    TESTLOG("1: Just before calling __nnmusl_get_tls_dtors_status");
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before calling nn::os::StartThread we expected 0 uncalled Destructor but we got %ld.", num_dtrs_to_call);

    TESTLOG("begin starting threads");
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors before starting threads.", num_dtrs_to_call);
    // スレッドの実行を開始する
    for (i = 0; i < NUM_THREADS; i++) {
        nn::os::StartThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX started", i, (size_t)&(g_Thread[i]));
    }

    TESTLOG("begin waiting threads");
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors before waiting threads.", num_dtrs_to_call);
    // スレッドが終了するのを待つ
    for (i = 0; i < NUM_THREADS; i++) {
        nn::os::WaitThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX did wait", i, (size_t)&(g_Thread[i]));
    }

    TESTLOG("begin destroying threads");
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors before destroying threads.", num_dtrs_to_call);
    // スレッドを破棄する
    for (i = 0; i < NUM_THREADS; i++) {
        nn::os::DestroyThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX destroyed", i, (size_t)&(g_Thread[i]));
    }

    TESTLOG("done destroying threads");
    TESTLOG("num_InnerHogeDestructorCalls = %d", num_InnerHogeDestructorCalls);
    TESTCASE_MESSAGE(num_InnerHogeDestructorCalls == NUM_THREADS, "Before exiting nnMain we expected %d calls to ~InnerHoge() but we got %ld.",
        NUM_THREADS, num_InnerHogeDestructorCalls);
    TESTLOG("num_HogeDestructorCalls = %d", num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == NUM_THREADS, "Before exiting nnMain we expected %d calls to ~Hoge() but we got %ld.",
        NUM_THREADS, num_HogeDestructorCalls);

    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTLOG("There are %ld registered destructors at the end of nnMain.", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before exiting nnMain we expected 0 uncalled Destructors but we got %ld.", num_dtrs_to_call);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

