﻿//
// File:       terrain_simulator.h
//
// Abstract:   This example shows how OpenCL can be used to create a procedural field of
//             grass on a generated terrain model which is then rendered with OpenGL.
//             Because OpenGL buffers are shared with OpenCL, the data can remain on the
//             graphics card, thus eliminating the API overhead of creating and submitting
//             the vertices from the host.
//
//             All geometry is generated on the compute device, and outputted into
//             a shared OpenGL buffer.  The terrain gets generated only within the
//             visible arc covering the camera's view frustum to avoid the need for
//             culling.  A page of grass is computed on the surface of the terrain as
//             bezier patches, and flow noise is applied to the angle of the blades
//             to simulate wind.  Multiple instances of grass are rendered at jittered
//             offsets to add more grass coverage without having to compute new pages.
//             Finally, a physically based sky shader (via OpenGL) is applied to
//             the background to provide an environment for the grass.
//
// Version:    <1.0>
//
// Disclaimer: IMPORTANT:  This Apple software is supplied to you by Apple Inc. ("Apple")
//             in consideration of your agreement to the following terms, and your use,
//             installation, modification or redistribution of this Apple software
//             constitutes acceptance of these terms.  If you do not agree with these
//             terms, please do not use, install, modify or redistribute this Apple
//             software.
//
//             In consideration of your agreement to abide by the following terms, and
//             subject to these terms, Apple grants you a personal, non - exclusive
//             license, under Apple's copyrights in this original Apple software ( the
//             "Apple Software" ), to use, reproduce, modify and redistribute the Apple
//             Software, with or without modifications, in source and / or binary forms;
//             provided that if you redistribute the Apple Software in its entirety and
//             without modifications, you must retain this notice and the following text
//             and disclaimers in all such redistributions of the Apple Software. Neither
//             the name, trademarks, service marks or logos of Apple Inc. may be used to
//             endorse or promote products derived from the Apple Software without specific
//             prior written permission from Apple.  Except as expressly stated in this
//             notice, no other rights or licenses, express or implied, are granted by
//             Apple herein, including but not limited to any patent rights that may be
//             infringed by your derivative works or by other works in which the Apple
//             Software may be incorporated.
//
//             The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
//             WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
//             WARRANTIES OF NON - INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A
//             PARTICULAR PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION
//             ALONE OR IN COMBINATION WITH YOUR PRODUCTS.
//
//             IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
//             CONSEQUENTIAL DAMAGES ( INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
//             SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//             INTERRUPTION ) ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION
//             AND / OR DISTRIBUTION OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER
//             UNDER THEORY OF CONTRACT, TORT ( INCLUDING NEGLIGENCE ), STRICT LIABILITY OR
//             OTHERWISE, EVEN IF APPLE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Copyright ( C ) 2008 Apple Inc. All Rights Reserved.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma once

/////////////////////////////////////////////////////////////////////////////

typedef struct _FixedTerrainUniforms
{
    int4 P[512];
    float4 G[64];
} FixedTerrainUniforms;

typedef struct _VariableTerrainUniforms
{
    float4 camera_position;
    float4 camera_rotation;
    float4 camera_view;
    float4 camera_left;
    int2 grid_resolution;
    float2 clip_range;
    float camera_fov;
    int vertex_count;
    int vertex_output_offset;
    int normal_output_offset;
    int texcoords_output_offset;
} VariableTerrainUniforms;

extern const int P[];
extern const float G[];

class TerrainSimulator
{

public:

    TerrainSimulator();
    ~TerrainSimulator();

    void finalize();
    bool setup( uint uiSizeX, uint uiSizeY, const char* pFilename );
    bool compute();

public:

    void setClipRange( float2 fV )
    {
        m_kClipRange = fV;
    }

    void setProjectedCorners(const float4 afCorners[4]);
    void setRangeMatrix(const float16 &rkM)
    {
        m_kRangeMatrix = rkM;
    }

    void setInverseModelViewProjectionMatrix(const float16 &rkM)
    {
        m_kInverseModelViewProjectionMatrix = rkM;
    }

    void setFalloffDistance(float fV)
    {
        m_fFalloff = fV;
    }

    void setCameraFov(float fV)
    {
        m_fCameraFov = fV;
    }

    void setCameraPosition(float4 fV)
    {
        m_kCameraPosition = fV;
    }

    void setCameraRotation(float4 fV)
    {
        m_kCameraRotation = fV;
    }

    void setCameraFrame(float4 fUp, float4 fView, float4 fLeft)
    {
        m_kCameraUp = fUp;
        m_kCameraView = fView;
        m_kCameraLeft = fLeft;
    }

    void setNoiseBias(float2 fV)
    {
        m_kNoiseBias = fV;
    }

    void setNoiseScale(float2 fV)
    {
        m_kNoiseScale = fV;
    }

    void setNoiseAmplitude(float fV)
    {
        m_fNoiseAmplitude = fV;
    }

    void setProjectedRange(float4 fV)
    {
        m_kProjectedRange = fV;
    }


public:

    float getFalloffDistance()
    {
        return m_fFalloff;
    }

    const float2& getClipRange()
    {
        return m_kClipRange;
    }

    float getCameraFov()
    {
        return m_fCameraFov;
    }

    const float2& getNoiseBias()
    {
        return m_kNoiseBias;
    }

    const float2& getNoiseScale()
    {
        return m_kNoiseScale;
    }

    float getNoiseAmplitude()
    {
        return m_fNoiseAmplitude;
    }

    void setExportBuffer( nn::gfx::GpuAddress address, size_t size )
    {
        m_ExportBufferAddress = address;
        m_ExportBufferSize = size;
    }

    uint getVertexCount()
    {
        return m_uiVertexCount;
    }

    uint getVertexComponentCount()
    {
        return m_uiVertexComponents;
    }

    uint getNormalCount()
    {
        return m_uiNormalCount;
    }

    uint getNormalComponentCount()
    {
        return m_uiNormalComponents;
    }

    uint getTexCoordCount()
    {
        return m_uiTexCoordCount;
    }

    uint getTexCoordComponentCount()
    {
        return m_uiTexCoordComponents;
    }


    uint getRequiredVertexBufferSize(uint uiSizeX, uint uiSizeZ);
    uint getRequiredNormalBufferSize(uint uiSizeX, uint uiSizeZ);
    uint getRequiredTexCoordBufferSize(uint uiSizeX, uint uiSizeZ);

protected:

    virtual bool allocate(uint uiSizeX, uint uiSizeY);

protected:

    int m_auiLocalDim[3];
    uint m_auiGlobalDim[3];

    uint m_uiSizeX;
    uint m_uiSizeY;

    float m_fJitterAmount;
    float m_fCameraFov;
    float4 m_kCameraPosition;
    float4 m_kCameraRotation;
    float4 m_kCameraUp;
    float4 m_kCameraView;
    float4 m_kCameraLeft;

    float4 m_kProjectedRange;
    float4 m_akProjectedCorners[4];

    float2 m_kNoiseBias;
    float2 m_kNoiseScale;
    float m_fNoiseAmplitude;

    uint m_uiVertexCount;
    uint m_uiVertexComponents;
    uint m_uiNormalCount;
    uint m_uiNormalComponents;
    uint m_uiTexCoordCount;
    uint m_uiTexCoordComponents;

    float2 m_kClipRange;
    float m_fFalloff;

    float16 m_kRangeMatrix;
    float16 m_kInverseModelViewProjectionMatrix;

    // Compute dispatch
    uint *m_auiDispatch;
    DEMOGfxBuffer m_DispatchBuf;
    int m_iVUniformsLoc;
    int m_iFUniformsLoc;
    int m_iExpBufLoc;

    DEMOGfxPipeline m_Pipeline;
    DEMOGfxBuffer m_VariableTerrainUniforms;
    DEMOGfxBuffer m_FixedTerrainUniforms;
    nn::gfx::GpuAddress m_ExportBufferAddress;
    size_t m_ExportBufferSize;
};

/////////////////////////////////////////////////////////////////////////////
