﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GX
#define MTX_USE_PS
#include <cafe/mat.h>
#endif

#include "compute_types.h"
#include "sky_renderer.h"
#include "camera.h"

#define odd(x) (x%2)

static const int MODEL_VTX_STRIDE = sizeof(f32) * 4;

struct VUniforms
{
    float16 m_kInvProjectionMatrix;
    float16 m_kInvModelviewMatrix;
};

struct PUniforms
{
    float4 m_kCameraDirection;
    float4 m_fExposure;
    float4 m_fSunAzimuth;
};

SkyRenderer::SkyRenderer() :
    m_afVertexData(0),
    m_kCameraDirection(0,0,0,0),
    m_fExposure(0,0,0,0),
    m_fSunAzimuth(0,0,0,0),
    m_iVUniformsLoc(0),
    m_iPUniformsLoc(0)
{
    // EMPTY!
}

SkyRenderer::~SkyRenderer()
{
    // EMPTY!
}

void SkyRenderer::finalize()
{
    m_Pipeline.Finalize();
    m_PixelUniformBuffer.Finalize();
    m_VertexBuffer.Finalize();
    m_VertexUniformBuffer.Finalize();
}

void SkyRenderer::setup( nn::gfx::PrimitiveTopology topology, const char* pFilename )
{
    m_Topology = topology;

    m_Pipeline.SetDefaults();
    DEMOGfxLoadShadersFromFile( &m_Pipeline.shaders, 0, pFilename );

    // Vertex buffer
    m_VertexBuffer.Initialize( sizeof( float4 ) * 4, NULL, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_afVertexData = m_VertexBuffer.Map< float4 >();
    m_afVertexData[ 0 ] = make_float4( -1.0f, -1.0f, 1.0f, 1.0f );
    m_afVertexData[ 1 ] = make_float4( -1.0f, 1.0f, 1.0f, 1.0f );
    m_afVertexData[ 2 ] = make_float4( 1.0f, -1.0f, 1.0f, 1.0f );
    m_afVertexData[ 3 ] = make_float4( 1.0f, 1.0f, 1.0f, 1.0f );
    m_VertexBuffer.Unmap();

    // Uniform Location lookup
    m_iVUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_VUniforms" );
    m_iPUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_PUniforms" );
    DEMOAssert( m_iVUniformsLoc != 0xffffffff && "u_VUniforms location is invalid." );
    DEMOAssert( m_iPUniformsLoc != 0xffffffff && "u_PUniforms location is invalid." );

    // Uniform buffers
    m_VertexUniformBuffer.Initialize( sizeof( struct VUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_PixelUniformBuffer.Initialize( sizeof( struct PUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );

    DEMOGfxInitShaderAttribute( &m_Pipeline.shaders, "a_position", 0, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &m_Pipeline.shaders, 0, MODEL_VTX_STRIDE, 0 );

    // Set Color Mask
    m_Pipeline.blendTargetStateCount = 1;
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( false );
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetChannelMask( nn::gfx::ChannelMask_All );

    // Color targets
    m_Pipeline.colorTargetStateCount = 1;
    m_Pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    m_Pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    m_Pipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );
    m_Pipeline.depthStencilStateInfo.SetDepthTestEnabled( false );

    m_Pipeline.rasterizerStateInfo.SetMultisampleEnabled( DEMOColorBufferInfo.GetMultisampleCount() > 1 );
    m_Pipeline.rasterizerStateInfo.EditMultisampleStateInfo().SetSampleCount( DEMOColorBufferInfo.GetMultisampleCount() );

    m_Pipeline.Initialize( &DEMODevice );

}

bool SkyRenderer::render( )
{
    dglPushMatrix();

    struct VUniforms* pVUniforms = m_VertexUniformBuffer.Map< struct VUniforms >( );
    memcpy( &pVUniforms->m_kInvModelviewMatrix , &m_kInvModelviewMatrix, sizeof( float16 ) );
    memcpy( &pVUniforms->m_kInvProjectionMatrix , &m_kInvProjectionMatrix, sizeof( float16 ) );
#ifdef CAFE
    GX2EndianSwap( pVUniforms, sizeof( struct VUniforms ) );
#endif
    m_VertexUniformBuffer.Unmap( );

    struct PUniforms* pPUniforms = m_PixelUniformBuffer.Map< struct PUniforms >( );
    memcpy( &pPUniforms->m_kCameraDirection , &m_kCameraDirection, sizeof( float4 ) );
    memcpy( &pPUniforms->m_fExposure , &m_fExposure, sizeof( float4 ) );
    memcpy( &pPUniforms->m_fSunAzimuth , &m_fSunAzimuth, sizeof( float4 ) );
#ifdef CAFE
    GX2EndianSwap( pPUniforms, sizeof( struct PUniforms ) );
#endif
    m_PixelUniformBuffer.Unmap( );

    DEMOCommandBuffer.SetVertexBuffer( 0, m_VertexBuffer.gpuAddress, MODEL_VTX_STRIDE, m_VertexBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( m_iVUniformsLoc, nn::gfx::ShaderStage_Vertex, m_VertexUniformBuffer.gpuAddress, m_VertexUniformBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( m_iPUniformsLoc, nn::gfx::ShaderStage_Pixel, m_PixelUniformBuffer.gpuAddress, m_PixelUniformBuffer.size );
    DEMOCommandBuffer.SetPipeline( &m_Pipeline.pipeline );
    DEMOCommandBuffer.Draw( m_Topology, 4, 0 );

    dglPopMatrix();
    return true;
}
