/*-----------------------------------------------------------------------*
  Project: csGrass
  File: sky.ps

  Copyright (C)2013 Nintendo  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo and/or its licensed developers
  and are protected by national and international copyright laws. They
  may not be disclosed to third parties or copied or duplicated in any
  form, in whole or in part, without the prior written consent of
  Nintendo.

  The content herein is highly confidential and should be handled
  accordingly.
*-----------------------------------------------------------------------*/

struct PixelShaderInput
{
    float4 position : SV_POSITION;
    float4 viewDir : viewDir;
};

cbuffer u_PUniforms : register( b0 )
{
    float4 CameraDirection;
    float4 Exposure;
    float4 SunAzimuth;
};

static const float4 SunDirection = float4( 0.0, 300.0, -100.0, 1.0 );
static const float4 SunIntensity = float4( 10.0, 10.0, 10.0, 1.0 );
static const float AtmosphereLengthBias = 1.0;

static const float AirZenith = 8400.0;
static const float AirHazeRatio = ( 1.25 / 8.4 );
static const float OuterRadius = ( 6378000.0 + AirZenith );
static const float InnerRadius = 6378000.0;

static const float4 RaleighCoefficients = float4( 4.1e-06, 6.93327e-06, 1.43768e-05, 1.0 );
static const float4 MieCoefficients = float4( 2.3e-06, 2.3e-06, 2.3e-06, 1.0 );
static const float Eccentricity = -0.994;

static const float PI = 3.14159265358979323846;

void AerialPerspective(
    float3 pos,
    float3 camera,
    float3 sun_dir,
    out float3 extinction,
    out float3 scatter,
    out float3 sun_color)
{
    // force viewpoint close to the ground to avoid color shifts
    camera.z = min(camera.z, InnerRadius + (OuterRadius - InnerRadius) * 0.25);

    float s_air  = length(pos - camera);
    float s_haze = s_air * AirHazeRatio;

    float3 extinction_air_haze = exp(-s_haze * (MieCoefficients.xyz + RaleighCoefficients.xyz));
    float3 extinction_air = exp(-(s_air - s_haze) * RaleighCoefficients.xyz);

    float3 view_dir = normalize(pos - camera);
    float cos_theta = dot(sun_dir, view_dir);

    float3 raleigh_theta = (1.0 + cos_theta * cos_theta) * RaleighCoefficients.xyz * 3.0 / (16.0 * PI);

    float3 mie_theta = 1.0 /(4.0 * PI) * MieCoefficients.xyz * (1.0 - (Eccentricity * Eccentricity)) *
                    pow(1.0 + (Eccentricity * Eccentricity) - (2.0 * Eccentricity * cos_theta), -3.0 / 2.0);

    float cos_theta_sun = -sun_dir.z;
    float theta_sun = (180.0/PI) * acos(cos_theta_sun);

    float t_air = AirZenith / (cos_theta_sun + 0.15 * pow(93.885 - theta_sun, -1.253));
    float t_haze = t_air * AirHazeRatio;
    
    sun_color = exp(-(RaleighCoefficients.xyz * t_air + MieCoefficients.xyz * t_haze));

    scatter = sun_color * SunIntensity.xyz *
              (((raleigh_theta + mie_theta) / (RaleighCoefficients.xyz + MieCoefficients.xyz)) * (1.0 - extinction_air_haze) +
              (raleigh_theta / RaleighCoefficients.xyz) * (1.0 - extinction_air) * extinction_air_haze);
    
    extinction = extinction_air * extinction_air_haze;
}

float3 ToneMap(float3 light)
{
    return (1.0 - exp(-light * Exposure.x));
}

float3 SkyColorFromAtmosphere(
    float3 pos, 
    float3 camera, 
    float3 sun_dir)
{
    float3 extinction, scatter, sun_color;
    AerialPerspective(pos, camera, sun_dir, extinction, scatter, sun_color);
    return ToneMap(scatter);
}

float4 main( PixelShaderInput input ) : SV_Target
{
    float3 cam_dir = normalize(float3(-CameraDirection.x, -CameraDirection.z, CameraDirection.y));
    float3 view_dir  = cam_dir + normalize(input.viewDir.xzy);

    float  sin_phi = view_dir.z;
    float  l = (-InnerRadius * sin_phi);
    l += sqrt( (InnerRadius * InnerRadius) * ((sin_phi * sin_phi) - 1.0) + OuterRadius * OuterRadius);
    
    float3 aerial_dir = normalize(float3(SunDirection.x, SunAzimuth.x, SunDirection.z));
    float3 pos = float3(0.0, 0.0, InnerRadius) + l * view_dir;
    float4 color = float4(SkyColorFromAtmosphere(pos, float3(0.0, 0.0, InnerRadius), aerial_dir), 1.0);

    return color;
}

