﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
  *--------------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
     #include <errno.h>
*/

#include "ntd-tests/ntd-tests.h"


static void test_get_schedparam(const char *src_file, int src_line,
                                pthread_t this_thread, int expect_policy,
                                int expect_priority)
{
    int rc = 0;
    struct sched_param param;
    int policy = -1;


    rc = pthread_getschedparam(this_thread, &policy, &param);

    NTD_TESTCASE_MESSAGE(rc == 0, src_file, src_line, "rc == 0",
        "pthread_getschedparam() returned %d", rc);

    NTD_TESTCASE_MESSAGE(param.sched_priority == expect_priority, src_file, src_line, "rc == param.sched_priority == expect_priority",
        "param.sched_priority is %d expected %d",
        param.sched_priority, expect_priority);

    TESTCASE_MESSAGE(policy == expect_policy,
        "policy is %d expected %d",
        policy, expect_policy);

}

#define TEST_GET_SCHEDPARAM(t, policy, priority) test_get_schedparam(__FILE__, __LINE__, t, policy, priority)

static void set_and_get_schedparam(const char *src_file, int src_line,
                                   pthread_t this_thread, int expect_policy,
                                   int expect_priority)
{
    int rc = 0;
    struct sched_param param;
    int policy = expect_policy;

    param.sched_priority = expect_priority;
    rc = pthread_setschedparam(this_thread, policy, &param);
    NTD_TESTCASE_MESSAGE(rc == 0, src_file, src_line, "rc == 0",
        "pthread_setschedparam() returned %d", rc);

    test_get_schedparam(src_file, src_line, this_thread, expect_policy, expect_priority);

}
#define TEST_SET_SCHEDPARAM(t, policy, priority) set_and_get_schedparam(__FILE__, __LINE__, t, policy, priority)

static void *test_threadFunc(void *arg)
{
    pthread_t this_thread = pthread_self();

    TEST_GET_SCHEDPARAM(this_thread, SCHED_OTHER, 0);

#if __NX__
    TEST_SET_SCHEDPARAM(this_thread, SCHED_OTHER, 4);
    TEST_SET_SCHEDPARAM(this_thread, SCHED_OTHER, -4);
#else
    /* can't set priority to anything but 0 on linux unless using real time scheduler */
    TEST_SET_SCHEDPARAM(this_thread, SCHED_OTHER, 0);
#endif

    return NULL;
}

void test_pthread_getschedparam()
{
    pthread_attr_t pta;
    int policy;
    struct sched_param param;
    pthread_t thread;
    int rc = 0;

    NTD_TEST_GROUP_START("getschedparam", 3);

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d", rc);

    rc = pthread_attr_setinheritsched(&pta, PTHREAD_EXPLICIT_SCHED);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setinheritsched() returned %d", rc);

    param.sched_priority = 0;
    rc = pthread_attr_setschedparam(&pta, &param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedparam() returned %d", rc);

    rc = pthread_create(&thread, &pta, test_threadFunc, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d", rc);

    rc = pthread_join(thread, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d", rc);

    rc = pthread_attr_destroy(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d", rc);

#ifdef __NX__
    // can't do this on linux as the thread has been deallocated

    // Make sure pthread_getschedparam() returns ESRCH when attempting to call
    // it after the thread has exited
    policy = -1;
    param.sched_priority = -1;
    rc = pthread_getschedparam(thread, &policy, &param);
    TESTCASE_MESSAGE(rc == ESRCH,
            "Called pthread_getschedparam() after thread exited. Expected return code to be 'ESRCH'(%d). Got  %d",
            ESRCH, rc);
#endif

    NTD_TEST_GROUP_END("getschedparam", 3);
}
