﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
     #include <string.h>
*/
#include <pthread.h>
#include <sched.h>
#include <inttypes.h>
#if defined(__NX__)
  #include <nnc/os.h>
  #include <nnc/os/os_ThreadApi.h>
#endif /* __NX__ */


#define BUMP_PRIO 1
#define REDUCE_PRIO -2
static pthread_t thread1, thread2, base_thread;
static pthread_attr_t pta1, pta2, base_pta;
static int base_priority = 0, base_inheritsched = 0;

static void validate_sched_param(pthread_attr_t *a, int current_prio)
{
    int                 rc=0;
    struct sched_param  param;
    int                 inheritsched;
    bool                test;


    rc = pthread_attr_getschedparam(a, &param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getschedparam() returned %d expected 0", rc);

    rc = pthread_attr_getinheritsched(a, &inheritsched);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getinheritsched() returned %d expected 0", rc);

    if (inheritsched == PTHREAD_EXPLICIT_SCHED) {
        TESTCASE_MESSAGE(param.sched_priority == current_prio,
            "PTHREAD_EXPLICIT_SCHED set: expected priority: %d, but current priority: %d (this is an error)\n",
            param.sched_priority, current_prio);
    } else if (inheritsched == PTHREAD_INHERIT_SCHED) {
        TESTCASE_MESSAGE(base_priority == current_prio,
            "PTHREAD_INHERIT_SCHED set: requested priority: %d should be ignored; base priority %d should equal current priority %d but doesn't (this is an error)\n",
            param.sched_priority, base_priority, current_prio);
    } else {
        TESTCASE_MESSAGE(0, "Unexpected inheritance %d\n", inheritsched);
    }
}

static void *threadfunc2(void *arg)
{
    pthread_t           thread = pthread_self();
    int                 priority;
    int                 arg_int = (int)arg;
    bool                test;

#if defined(__NX__)
    priority = nnosGetThreadPriority((nnosThreadType*)thread) - NN_OS_DEFAULT_THREAD_PRIORITY;
    TESTWARN("Inside secondary thread: thread with assigned priority %p\n", arg_int);
    if (arg_int == -2) {
        TESTCASE_MESSAGE(thread == thread1,
            "thread1 %p is expected but got %p", thread1, thread);
        validate_sched_param(&pta1, priority);
    } else{
        test = (thread == thread2);
        TESTCASE_MESSAGE(thread == thread2,
            "thread2 %p is expected but got %p", thread2, thread);
        validate_sched_param(&pta2, priority);
    }
#endif

    return NULL;
}

static void *threadfunc1(void *arg)
{
    int                   rc=0;
    struct sched_param    param;

    TESTWARN("Inside secondary thread: thread %p\n", arg);

#if defined(__NX__)
    base_priority = nnosGetThreadPriority((nnosThreadType*)pthread_self()) - NN_OS_DEFAULT_THREAD_PRIORITY;
    TESTWARN("Inside threadfunc1: base thread actually has priority %d\n", base_priority);
#endif

    TESTWARN("Create a thread attributes object\n");
    rc = pthread_attr_init(&pta1);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    memset(&param, 0, sizeof(param));
#if defined(__NX__)
    param.sched_priority = -2;

    TESTWARN("Setting scheduling priority to -2\n");
#else
    param.sched_priority = 0;

    TESTWARN("LINUX does not allow -2 for priority with default scheduler\n");
    TESTWARN("Setting scheduling priority to 0\n");

#endif
    rc = pthread_attr_setschedparam(&pta1, &param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedparam() 1 returned %d expected 0", rc);

    TESTWARN("Setting scheduling inherit to PTHREAD_EXPLICIT_SCHED\n");
    rc = pthread_attr_setinheritsched(&pta1, PTHREAD_EXPLICIT_SCHED);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setinheritsched() returned %d expected 0", rc);

    rc = pthread_create(&thread1, &pta1, threadfunc2, (void *)(-2));
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);

    TESTWARN("Wait for the thread to exit 1\n");
    rc = pthread_join(thread1, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);

    TESTWARN("\nCreate a thread attributes object\n");
    rc = pthread_attr_init(&pta2);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    memset(&param, 0, sizeof(param));
 #if defined(__NX__)
    param.sched_priority = 2;

    TESTWARN("Setting scheduling priority to 2\n");
#else
    param.sched_priority = 0;

    TESTWARN("LINUX does not allow 2 for priority with default scheduler\n");
    TESTWARN("Setting scheduling priority to 0\n");

#endif
    rc = pthread_attr_setschedparam(&pta2, &param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedparam() returned %d expected 0", rc);

    TESTWARN("Setting scheduling inherit to PTHREAD_INHERIT_SCHED\n");
    rc = pthread_attr_setinheritsched(&pta2, PTHREAD_INHERIT_SCHED);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setinheritsched() returned %d expected 0", rc);

    TESTWARN("Create thread using attributes that allow join\n");
    rc = pthread_create(&thread2, &pta2, threadfunc2, (void *)(2));
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);

    TESTWARN("Wait for the thread to exit 2\n");
    rc = pthread_join(thread2, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);

    rc = pthread_attr_destroy(&pta1);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    rc = pthread_attr_destroy(&pta2);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    return NULL;
}


void test_pthread_attr_setschedparam_2()
{
    int                   rc=0;
    struct sched_param    param;
#if defined(__NX__)
    int                   main_prio;
#endif

    NTD_TEST_GROUP_START("attr_setschedparam_2", 3);

#if defined(__NX__)
    main_prio = nnosGetThreadPriority((nnosThreadType*)pthread_self()) - NN_OS_DEFAULT_THREAD_PRIORITY;
    TESTWARN("Inside test_pthread_attr_setschedparam_2: main thread actually has priority %d\n", main_prio);
#endif

    TESTWARN("Create a thread attributes object\n");
    rc = pthread_attr_init(&base_pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    /* priority and inheritance are the default */
    rc = pthread_attr_getschedparam(&base_pta, &param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getschedparam() returned %d expected 0", rc);
    TESTWARN("base_priority is %d\n", param.sched_priority);

    rc = pthread_attr_getinheritsched(&base_pta, &base_inheritsched);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getinheritsched() returned %d expected 0", rc);
    TESTWARN("base_inheritsched is %d\n", base_inheritsched);

    rc = pthread_create(&base_thread, &base_pta, threadfunc1, (void *)0);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);

    TESTWARN("Wait for the thread to exit\n");
    rc = pthread_join(base_thread, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);

    TESTWARN("Destroy thread attributes object\n");
    rc = pthread_attr_destroy(&base_pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    NTD_TEST_GROUP_END("attr_setschedparam_2", 3);
}

#if 0
Debug Output:

Enter Testcase - QP0WTEST/TASSP0
Create a thread attributes object
Get scheduling parameters
The thread attributes object indicates priority: 0
Setting scheduling parameters
Get scheduling parameters
The thread attributes object indicates priority: 0
Destroy thread attributes object
Main completed
#endif
